import csv
import sqlite3
import datetime
import pathlib
import tkinter as tk
from tkinter import ttk, messagebox, simpledialog
from contextlib import contextmanager
from typing import Optional

# -------------------- Util --------------------
@contextmanager
def _tx(conn: sqlite3.Connection):
    """Helper de transação (BEGIN/COMMIT/ROLLBACK)."""
    try:
        conn.execute("BEGIN")
        yield
        conn.execute("COMMIT")
    except Exception:
        try:
            conn.execute("ROLLBACK")
        except Exception:
            pass
        raise

def _has_status(conn: sqlite3.Connection) -> bool:
    """Verifica se a tabela sales possui coluna 'status'."""
    try:
        for r in conn.execute("PRAGMA table_info(sales)"):
            if (r[1] or "").lower() == "status":
                return True
    except Exception:
        pass
    return False

def _table_exists(conn: sqlite3.Connection, table: str) -> bool:
    try:
        cur = conn.execute("SELECT name FROM sqlite_master WHERE type='table' AND name=?", (table,))
        return cur.fetchone() is not None
    except Exception:
        return False

def _row_get(row: sqlite3.Row, key: str, default=""):
    try:
        return row[key]
    except Exception:
        return default

def _fmt_money(v) -> str:
    try:
        return ("R$ {0:,.2f}".format(float(v))).replace(",", "X").replace(".", ",").replace("X", ".")
    except Exception:
        return ""

def _fmt_dt(iso_str: Optional[str]) -> str:
    """Converte 'YYYY-MM-DDTHH:MM:SS(.micro)' para 'DD/MM/YYYY HH:MM'."""
    if not iso_str:
        return ""
    try:
        dt = datetime.datetime.fromisoformat(str(iso_str).replace("Z", ""))
        return dt.strftime("%d/%m/%Y %H:%M")
    except Exception:
        return str(iso_str)

SQL_PM = (
    "SELECT payment_method, COUNT(*), ROUND(SUM(total_gross),2) "
    "FROM sales WHERE ts >= ? AND ts <= ? "
    "GROUP BY payment_method ORDER BY payment_method"
)
SQL_ST = (
    "SELECT COALESCE(status,'<sem_status>'), ROUND(SUM(total_gross),2) "
    "FROM sales WHERE ts >= ? AND ts <= ? GROUP BY status"
)
SQL_MV = (
    "SELECT created_at, type, method, amount, reason, "
    "COALESCE(reference_type,''), COALESCE(reference_id,''), COALESCE(notes,'') "
    "FROM cash_movements WHERE created_at >= ? AND created_at <= ? ORDER BY created_at"
)

# -------------------- Janela do Caixa --------------------
class CaixaWindow:
    def __init__(self, app, get_conn):
        self.app = app
        self.get_conn = get_conn

        # ---- Janela ----
        self.win = tk.Toplevel(app.root)
        self.win.title("Caixa")
        self.win.geometry("900x640")
        self.win.transient(app.root)
        self.win.grab_set()
        self.win.minsize(860, 600)

        # ---- Estilos ----
        self._setup_styles()

        # ===================== TOP BAR (título + status + [-][□][X]) =====================
        top = ttk.Frame(self.win, padding=(14, 10, 14, 6))
        top.pack(fill="x")

        title = ttk.Label(top, text="Abertura & Fechamento de Caixa", style="Title.TLabel")
        title.pack(side="left")

        spacer = ttk.Label(top, text=" ")
        spacer.pack(side="left", expand=True)

        self.lbl_status = ttk.Label(top, text="Status do Caixa: —", style="Status.Bad.TLabel")
        self.lbl_status.pack(side="left", padx=(0, 12))

        # Botões de janela (minimizar, maximizar/restaurar, fechar)
        winbtns = ttk.Frame(top)
        winbtns.pack(side="right")

        self.btn_min = ttk.Button(winbtns, text="-", width=3, command=self._minimize)
        self.btn_max = ttk.Button(winbtns, text="□", width=3, command=self._toggle_maximize)
        self.btn_x   = ttk.Button(winbtns, text="X", width=3, command=self._close_window)

        self.btn_min.pack(side="left", padx=2)
        self.btn_max.pack(side="left", padx=2)
        self.btn_x.pack(side="left", padx=2)

        # ===================== TOOLBAR DE AÇÕES =====================
        toolbar = ttk.Frame(self.win, padding=(14, 0, 14, 8))
        toolbar.pack(fill="x")

        BW = 18  # largura em caracteres
        self.btn_supply   = ttk.Button(toolbar, text="Suprimento (F7)", width=BW, command=self.do_supply)
        self.btn_withdraw = ttk.Button(toolbar, text="Sangria (F6)",    width=BW, command=self.do_withdraw)
        self.btn_open     = ttk.Button(toolbar, text="Abrir (F2)",      width=BW, command=self.open_cash,  style="Primary.TButton")
        self.btn_close    = ttk.Button(toolbar, text="Fechar (F3)",     width=BW, command=self.close_cash)
        self.btn_refresh  = ttk.Button(toolbar, text="Atualizar (F5)",  width=BW, command=self.refresh)

        # alinha à direita, com espaçamento uniforme
        for b in (self.btn_supply, self.btn_withdraw, self.btn_open, self.btn_close, self.btn_refresh):
            b.pack(side="right", padx=6)

        # ===================== DETALHES =====================
        details = ttk.LabelFrame(self.win, text="Detalhes do Caixa", padding=(14, 10))
        details.pack(fill="x", padx=14, pady=(0, 8))

        self.vars = {k: tk.StringVar(value="") for k in (
            "id","opened_at","opening_amount","operator_open",
            "closed_at","closing_amount","operator_close","notes"
        )}

        grid = ttk.Frame(details)
        grid.pack(fill="x")

        def row(r, label, key):
            ttk.Label(grid, text=label, width=22, anchor="w").grid(row=r, column=0, sticky="w", pady=2)
            ttk.Label(grid, textvariable=self.vars[key], style="Value.TLabel").grid(row=r, column=1, sticky="w", pady=2)

        row(0, "ID:",                  "id")
        row(1, "Abertura em:",         "opened_at")
        row(2, "Valor abertura:",      "opening_amount")
        row(3, "Operador abertura:",   "operator_open")
        row(4, "Fechamento em:",       "closed_at")
        row(5, "Valor fechamento:",    "closing_amount")
        row(6, "Operador fechamento:", "operator_close")
        row(7, "Observações:",         "notes")

        # ===================== TOTAIS =====================
        totals = ttk.LabelFrame(self.win, text="Totais do Período do Caixa", padding=(12, 10))
        totals.pack(fill="both", expand=True, padx=14, pady=(0, 12))

        columns = ("forma", "qtd", "total")
        self.tv = ttk.Treeview(totals, columns=columns, show="headings", height=12)
        self.tv.heading("forma", text="Forma")
        self.tv.heading("qtd",   text="Qtd")
        self.tv.heading("total", text="Total (R$)")
        self.tv.column("forma", width=360, anchor="w")
        self.tv.column("qtd",   width=90,  anchor="center")
        self.tv.column("total", width=180, anchor="e")
        self.tv.pack(fill="both", expand=True, side="left")

        # zebra
        self.tv.tag_configure("odd",  background="#f3f3f3")
        self.tv.tag_configure("even", background="#ffffff")

        vsb = ttk.Scrollbar(totals, orient="vertical", command=self.tv.yview)
        vsb.pack(side="right", fill="y")
        self.tv.configure(yscrollcommand=vsb.set)

        # ---- Atalhos ----
        self.win.bind("<F2>", lambda e: self.open_cash())
        self.win.bind("<F3>", lambda e: self.close_cash())
        self.win.bind("<F5>", lambda e: self.refresh())
        self.win.bind("<F6>", lambda e: self.do_withdraw())
        self.win.bind("<F7>", lambda e: self.do_supply())

        # Carrega estado
        self.refresh()

    # -------------------- Estilos UI --------------------
    def _setup_styles(self):
        style = ttk.Style(self.win)
        try:
            style.theme_use("clam")
        except Exception:
            pass
        style.configure("Title.TLabel", font=("Segoe UI", 14, "bold"))
        style.configure("Value.TLabel", font=("Segoe UI", 10))
        style.configure("Primary.TButton", padding=(10, 6), font=("Segoe UI", 10, "bold"))
        style.configure("TButton", padding=(10, 6), font=("Segoe UI", 10))
        style.configure("Status.Good.TLabel", foreground="#0a7", font=("Segoe UI", 10, "bold"))
        style.configure("Status.Bad.TLabel",  foreground="#c00", font=("Segoe UI", 10, "bold"))

    # -------------------- Botões de Janela --------------------
    def _minimize(self):
        try:
            self.win.iconify()
        except Exception:
            pass

    def _toggle_maximize(self):
        try:
            # no Windows, state('zoomed') maximiza; repetir alterna
            current = str(self.win.state())
            if current == "zoomed":
                self.win.state("normal")
            else:
                self.win.state("zoomed")
        except Exception:
            pass

    def _close_window(self):
        try:
            self.win.destroy()
        except Exception:
            pass

    # -------------------- Helpers --------------------
    def _get_open(self):
        with self.get_conn() as con:
            con.row_factory = sqlite3.Row
            return con.execute("SELECT * FROM cash_registers WHERE closed_at IS NULL ORDER BY id DESC LIMIT 1").fetchone()

    def _get_last_any(self):
        with self.get_conn() as con:
            con.row_factory = sqlite3.Row
            return con.execute("SELECT * FROM cash_registers ORDER BY id DESC LIMIT 1").fetchone()

    def _period_bounds(self, opened_at: str, closed_at: Optional[str]):
        t0 = opened_at
        t1 = closed_at or datetime.datetime.now().isoformat()
        return t0, t1

    def _current_operator(self) -> str:
        try:
            return self.app.current_user["username"]
        except Exception:
            return "<desconhecido>"

    # -------------------- UI Actions --------------------
    def refresh(self):
        try:
            row = self._get_open()
            has_open = bool(row)
            if has_open:
                self.lbl_status.config(text="Status do Caixa: ABERTO", style="Status.Good.TLabel")
            else:
                row = self._get_last_any()
                self.lbl_status.config(text="Status do Caixa: FECHADO", style="Status.Bad.TLabel")

            self.btn_open.config(state=("disabled" if has_open else "normal"))
            self.btn_close.config(state=("normal" if has_open else "disabled"))
            self.btn_withdraw.config(state=("normal" if has_open else "disabled"))
            self.btn_supply.config(state=("normal" if has_open else "disabled"))

            if row:
                self.vars["id"].set(_row_get(row, "id"))
                self.vars["opened_at"].set(_fmt_dt(_row_get(row, "opened_at")))
                self.vars["opening_amount"].set(_fmt_money(_row_get(row, "opening_amount")))
                self.vars["operator_open"].set(_row_get(row, "operator_open") or "")
                self.vars["closed_at"].set(_fmt_dt(_row_get(row, "closed_at")))
                self.vars["closing_amount"].set(_fmt_money(_row_get(row, "closing_amount")))
                self.vars["operator_close"].set(_row_get(row, "operator_close") or "")
                self.vars["notes"].set(_row_get(row, "notes") or "")

                t0, t1 = self._period_bounds(_row_get(row, "opened_at"), _row_get(row, "closed_at") or None)
                self._load_totals(t0, t1)
            else:
                for k in self.vars:
                    self.vars[k].set("")
                for iid in self.tv.get_children():
                    self.tv.delete(iid)
        except Exception as e:
            messagebox.showerror("Caixa", "Falha ao atualizar: {0}".format(e))

    def _load_totals(self, t0: str, t1: str):
        for iid in self.tv.get_children():
            self.tv.delete(iid)
        try:
            with self.get_conn() as con:
                pm_rows = list(con.execute(SQL_PM, (t0, t1)))
                for idx, (forma, qtd, total) in enumerate(pm_rows):
                    tag = "even" if (idx % 2 == 0) else "odd"
                    self.tv.insert("", "end", values=(forma, qtd, _fmt_money(total)), tags=(tag,))
        except Exception as e:
            messagebox.showwarning("Caixa", "Não foi possível carregar totais: {0}".format(e))

    def open_cash(self):
        if self._get_open():
            messagebox.showwarning("Caixa", "Já existe um caixa aberto.")
            return
        amount = simpledialog.askfloat("Abrir caixa", "Valor em dinheiro na gaveta (opcional):", parent=self.win, minvalue=0.0)
        if amount is None:
            amount = 0.0
        notes = simpledialog.askstring("Abrir caixa", "Observações (opcional):", parent=self.win) or ""
        operator = self._current_operator()
        now_iso = datetime.datetime.now().isoformat()
        try:
            with self.get_conn() as con:
                with _tx(con):
                    con.execute(
                        "INSERT INTO cash_registers(opened_at, opening_amount, operator_open, notes) VALUES(?,?,?,?)",
                        (now_iso, float(amount), operator, notes),
                    )
            self.refresh()
            messagebox.showinfo("Caixa", "Caixa aberto com sucesso.")
        except Exception as e:
            messagebox.showerror("Caixa", "Falha ao abrir caixa: {0}".format(e))

    def close_cash(self):
        row = self._get_open()
        if not row:
            messagebox.showwarning("Caixa", "Não há caixa aberto.")
            return
        counted = simpledialog.askfloat("Fechar caixa", "Valor contado (dinheiro):", parent=self.win, minvalue=0.0)
        if counted is None:
            counted = 0.0
        t0 = _row_get(row, "opened_at")
        t1 = datetime.datetime.now().isoformat()
        operator_close = self._current_operator()

        # Totais para CSV
        try:
            with self.get_conn() as con:
                pm_rows = list(con.execute(SQL_PM, (t0, t1)))
                st_rows = []
                if _has_status(con):
                    st_rows = list(con.execute(SQL_ST, (t0, t1)))
                mov_rows = []
                if _table_exists(con, "cash_movements"):
                    mov_rows = list(con.execute(SQL_MV, (t0, t1)))
        except Exception as e:
            messagebox.showerror("Caixa", "Falha ao carregar totais para fechamento: {0}".format(e))
            return

        # Fecha caixa (UPDATE)
        try:
            with self.get_conn() as con:
                with _tx(con):
                    con.execute(
                        "UPDATE cash_registers SET closed_at=?, closing_amount=?, operator_close=? WHERE id=?",
                        (t1, float(counted), operator_close, _row_get(row, "id")),
                    )
        except Exception as e:
            messagebox.showerror("Caixa", "Falha ao fechar caixa: {0}".format(e))
            return

        # CSV automático
        out = pathlib.Path.cwd() / "fechamento_caixa_{0}_{1}.csv".format(_row_get(row, "id"), t1[:10])
        try:
            with out.open("w", newline="", encoding="utf-8") as f:
                w = csv.writer(f, delimiter=';')
                w.writerow(["Caixa", _row_get(row, "id")])
                w.writerow(["Abertura em", _fmt_dt(t0)])
                w.writerow(["Fechamento em", _fmt_dt(t1)])
                w.writerow(["Operador (abertura)", _row_get(row, "operator_open") or ""])
                w.writerow(["Operador (fechamento)", operator_close])
                w.writerow([])
                w.writerow(["Forma", "Qtd", "Total"])
                for pm, qtd, total in pm_rows:
                    w.writerow([pm, qtd, _fmt_money(total)])
                w.writerow([])
                w.writerow(["Total por status", "Total"])
                if st_rows:
                    for st, total in st_rows:
                        w.writerow([st, _fmt_money(total)])
                else:
                    w.writerow(["<sem_status>", "—"])
                if mov_rows:
                    w.writerow([])
                    w.writerow(["Movimentos de Caixa (se houver)"])
                    w.writerow(["Data/Hora", "Tipo", "Método", "Valor", "Razão", "Ref.Type", "Ref.ID", "Obs"])
                    for created_at, mtype, method, amount, reason, rtype, rid, obs in mov_rows:
                        w.writerow([_fmt_dt(created_at), mtype, method, _fmt_money(amount), reason, rtype, rid, obs])
        except Exception as e:
            messagebox.showwarning("Caixa", "Caixa fechado, mas não foi possível gravar o CSV:\n{0}".format(e))
        else:
            messagebox.showinfo("Caixa", "Caixa fechado.\nCSV gerado:\n{0}".format(out))

        self.refresh()

    # -------------------- Sangria / Suprimento --------------------
    def _insert_movement_if_possible(self, created_at: str, mtype: str, amount: float, reason: str, method: str, notes: str) -> bool:
        """Insere em cash_movements SE a tabela existir, caso contrário anexa em notes do caixa."""
        try:
            with self.get_conn() as con:
                if _table_exists(con, "cash_movements"):
                    with _tx(con):
                        con.execute(
                            "INSERT INTO cash_movements (created_at, type, method, amount, reason, reference_type, reference_id, notes) "
                            "VALUES(?,?,?,?,?,NULL,NULL,?)",
                            (created_at, mtype, method, float(amount), reason, notes or ""),
                        )
                else:
                    row = self._get_open()
                    if row:
                        with _tx(con):
                            old_notes = _row_get(row, "notes") or ""
                            add = "\n[{0}] {1}/{2} {3}: R$ {4:.2f} — {5}".format(created_at, mtype, method, reason, amount, (notes or ""))
                            con.execute("UPDATE cash_registers SET notes = ? WHERE id = ?", (old_notes + add, _row_get(row, "id")))
        except Exception as e:
            messagebox.showerror("Caixa", "Falha ao registrar movimento: {0}".format(e))
            return False
        return True

    def do_withdraw(self):
        if not self._get_open():
            messagebox.showwarning("Caixa", "Abra o caixa antes de registrar sangria.")
            return
        val = simpledialog.askfloat("Sangria", "Valor a retirar (R$):", parent=self.win, minvalue=0.01)
        if val is None:
            return
        obs = simpledialog.askstring("Sangria", "Motivo/observação:", parent=self.win) or ""
        ts = datetime.datetime.now().isoformat()
        if self._insert_movement_if_possible(ts, "OUT", float(val), "WITHDRAW", "CASH", obs):
            messagebox.showinfo("Caixa", "Sangria registrada.")
            self.refresh()

    def do_supply(self):
        if not self._get_open():
            messagebox.showwarning("Caixa", "Abra o caixa antes de registrar suprimento.")
            return
        val = simpledialog.askfloat("Suprimento", "Valor a adicionar (R$):", parent=self.win, minvalue=0.01)
        if val is None:
            return
        obs = simpledialog.askstring("Suprimento", "Motivo/observação:", parent=self.win) or ""
        ts = datetime.datetime.now().isoformat()
        if self._insert_movement_if_possible(ts, "IN", float(val), "SUPPLY", "CASH", obs):
            messagebox.showinfo("Caixa", "Suprimento registrado.")
            self.refresh()

# Factory
def open_caixa(app, get_conn):
    return CaixaWindow(app, get_conn)
