# config_nome_recibo.py
import tkinter as tk
from tkinter import ttk, messagebox
from typing import Callable
import sqlite3

# --- helpers locais para usar a tabela settings ---
def _ensure_settings(conn: sqlite3.Connection):
    conn.execute("""
        CREATE TABLE IF NOT EXISTS settings (
            key   TEXT PRIMARY KEY,
            value TEXT
        )
    """)

def _get_setting(conn: sqlite3.Connection, key: str, default: str = "") -> str:
    _ensure_settings(conn)
    row = conn.execute("SELECT value FROM settings WHERE key=?", (key,)).fetchone()
    return row[0] if row and row[0] is not None else default

def _set_setting(conn: sqlite3.Connection, key: str, value: str):
    _ensure_settings(conn)
    conn.execute("INSERT INTO settings(key,value) VALUES(?, ?) ON CONFLICT(key) DO UPDATE SET value=excluded.value",
                 (key, value))

# --- janela de configuração ---
def open_config_nome_recibo(parent_window: tk.Misc, get_conn: Callable[[], sqlite3.Connection]):
    win = tk.Toplevel(parent_window)
    win.title("Configurar Nome da Nota/Recibo")
    win.resizable(False, False)
    win.transient(parent_window)
    win.grab_set()
    try:
        win.attributes("-topmost", True)
    except Exception:
        pass

    # centralizar
    parent_window.update_idletasks()
    px, py = parent_window.winfo_rootx(), parent_window.winfo_rooty()
    pw, ph = parent_window.winfo_width(), parent_window.winfo_height()
    win.update_idletasks()
    ww, wh = 420, 180
    x = px + (pw - ww)//2
    y = py + (ph - wh)//2
    win.geometry(f"{ww}x{wh}+{max(x,0)}+{max(y,0)}")

    frm = ttk.Frame(win, padding=12)
    frm.pack(fill="both", expand=True)

    ttk.Label(frm, text="Nome (cabeçalho da Nota/Recibo):").grid(row=0, column=0, sticky="w")
    var_nome = tk.StringVar()
    ttk.Entry(frm, textvariable=var_nome, width=46).grid(row=1, column=0, columnspan=2, sticky="we", pady=(4,10))

    ttk.Label(frm, text="(opcional) Usar o mesmo nome no PIX:").grid(row=2, column=0, sticky="w", pady=(2,0))
    var_sync_pix = tk.BooleanVar(value=True)
    ttk.Checkbutton(frm, variable=var_sync_pix, text="Atualizar também o PIX_MERCHANT_NAME").grid(row=3, column=0, sticky="w")

    # carrega valores atuais
    with get_conn() as conn:
        atual = _get_setting(conn, "NFCE_FANTASIA", "") or _get_setting(conn, "PIX_MERCHANT_NAME", "")
    var_nome.set(atual)

    def salvar():
        nome = (var_nome.get() or "").strip()
        if not nome:
            messagebox.showerror("Configurar", "Informe um nome para o cabeçalho da Nota/Recibo.", parent=win)
            return
        # PIX tem limite de 25 chars; avisamos mas não bloqueamos o recibo
        if var_sync_pix.get() and len(nome) > 25:
            if not messagebox.askyesno("Configurar",
                "O nome do PIX tem limite de 25 caracteres.\nDeseja salvar mesmo assim (será cortado no QR)?",
                parent=win):
                return
        try:
            with get_conn() as conn:
                _set_setting(conn, "NFCE_FANTASIA", nome)
                if var_sync_pix.get():
                    _set_setting(conn, "PIX_MERCHANT_NAME", nome[:25])  # corta para o QR
            messagebox.showinfo("Configurar", "Nome salvo com sucesso.", parent=win)
            win.destroy()
        except Exception as e:
            messagebox.showerror("Configurar", f"Falha ao salvar: {e}", parent=win)

    btns = ttk.Frame(frm)
    btns.grid(row=4, column=0, sticky="e", pady=(14,0))
    ttk.Button(btns, text="Cancelar", command=win.destroy).pack(side="left", padx=(0,8))
    ttk.Button(btns, text="Salvar", command=salvar).pack(side="left")

    win.bind("<Return>", lambda e: salvar())
    win.bind("<Escape>", lambda e: win.destroy())
