# erp_pdv_tk.py 
# ERP/PDV Desktop — Tkinter + SQLite

import os
import sys
import sqlite3
import binascii
import hashlib
import secrets
from datetime import datetime
from contextlib import closing
from pathlib import Path

import tkinter as tk
from tkinter import ttk, messagebox, simpledialog

# Viewer opcional do Manual (não quebra se ausente)
try:
    from tools.manual_viewer import open_manual as _open_manual_viewer
except Exception:
    _open_manual_viewer = None

APP_TITLE = "FabricaDigital.shop — ERP/PDV (Tkinter + SQLite)"  # referência interna
DB_ENV = "ERP_PDV_DB"
DEFAULT_DB_NAME = "erp_pdv.db"

# ---- Compat: UTC sem DeprecationWarning no 3.13
try:
    from datetime import UTC  # Py 3.11+
except Exception:
    from datetime import timezone as _tz
    UTC = _tz.utc

def now_utc_iso() -> str:
    return datetime.now(UTC).isoformat()

# -----------------------------
# Util: Caminho do banco
# -----------------------------
def get_db_path() -> str:
    env = os.environ.get(DB_ENV)
    if env:
        return os.path.abspath(env)
    try:
        base = os.path.dirname(os.path.abspath(__file__))
    except Exception:
        base = os.getcwd()
    return os.path.join(base, DEFAULT_DB_NAME)

# -----------------------------
# Segurança (PBKDF2)
# -----------------------------
ALG = "pbkdf2_sha256"
ITERATIONS = 200_000
SALT_LEN = 16

def _pbkdf2(password: str, salt: bytes, iterations: int) -> bytes:
    try:
        return hashlib.pbkdf2_hmac("sha256", password.encode("utf-8"), salt, iterations)
    except AttributeError:
        from hmac import new as hmac_new
        dk = b""
        block = 1
        while len(dk) < 32:
            u = hmac_new(password.encode("utf-8"), salt + block.to_bytes(4, "big"), hashlib.sha256).digest()
            r = bytearray(u)
            for _ in range(iterations - 1):
                u = hmac_new(password.encode("utf-8"), u, hashlib.sha256).digest()
                r = bytearray(x ^ y for x, y in zip(r, u))
            dk += bytes(r)
            block += 1
        return dk[:32]

def make_password(password: str, iterations: int = ITERATIONS) -> str:
    salt = secrets.token_bytes(SALT_LEN)
    dk = _pbkdf2(password, salt, iterations)
    return f"{ALG}${iterations}${binascii.hexlify(salt).decode()}${binascii.hexlify(dk).decode()}"

def check_password(password: str, encoded: str) -> bool:
    try:
        alg, it, salt_hex, hash_hex = encoded.split("$", 3)
        if alg != ALG:
            return False
        it = int(it)
        salt = binascii.unhexlify(salt_hex)
        dk = _pbkdf2(password, salt, it)
        return secrets.compare_digest(binascii.hexlify(dk).decode(), hash_hex)
    except Exception:
        return False

# -----------------------------
# Conexão e schema
# -----------------------------
def connect(db_path: str):
    conn = sqlite3.connect(db_path, timeout=30, isolation_level=None)
    conn.row_factory = sqlite3.Row
    try:
        conn.execute("PRAGMA foreign_keys = ON")
    except Exception:
        pass
    return conn

def init_db(conn: sqlite3.Connection):
    # Users/Perms existentes
    conn.execute("""
    CREATE TABLE IF NOT EXISTS users(
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      username TEXT UNIQUE NOT NULL,
      full_name TEXT,
      password_hash TEXT NOT NULL,
      is_admin INTEGER NOT NULL DEFAULT 0,
      active INTEGER NOT NULL DEFAULT 1,
      must_change_password INTEGER NOT NULL DEFAULT 0,
      created_at TEXT NOT NULL
    )
    """)
    conn.execute("""
    CREATE TABLE IF NOT EXISTS permissions(
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      code TEXT UNIQUE NOT NULL,
      name TEXT NOT NULL
    )
    """)
    conn.execute("""
    CREATE TABLE IF NOT EXISTS user_permissions(
      user_id INTEGER NOT NULL,
      perm_id INTEGER NOT NULL,
      UNIQUE(user_id, perm_id),
      FOREIGN KEY(user_id) REFERENCES users(id) ON DELETE CASCADE,
      FOREIGN KEY(perm_id) REFERENCES permissions(id) ON DELETE CASCADE
    )
    """)
    conn.execute("""
    CREATE TABLE IF NOT EXISTS settings(
      key TEXT PRIMARY KEY,
      value TEXT
    )
    """)

    # --- NOVO: cargos/roles (compatível)
    conn.execute("""
    CREATE TABLE IF NOT EXISTS roles(
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      code TEXT UNIQUE NOT NULL,
      name TEXT NOT NULL
    )
    """)
    conn.execute("""
    CREATE TABLE IF NOT EXISTS role_permissions(
      role_id INTEGER NOT NULL,
      perm_id INTEGER NOT NULL,
      UNIQUE(role_id, perm_id),
      FOREIGN KEY(role_id) REFERENCES roles(id) ON DELETE CASCADE,
      FOREIGN KEY(perm_id) REFERENCES permissions(id) ON DELETE CASCADE
    )
    """)
    conn.execute("""
    CREATE TABLE IF NOT EXISTS user_roles(
      user_id INTEGER NOT NULL,
      role_id INTEGER NOT NULL,
      UNIQUE(user_id, role_id),
      FOREIGN KEY(user_id) REFERENCES users(id) ON DELETE CASCADE,
      FOREIGN KEY(role_id) REFERENCES roles(id) ON DELETE CASCADE
    )
    """)

    perms = [
        ("PDV", "Módulo de Vendas (PDV)"),
        ("ESTOQUE", "Módulo de Estoque"),
        ("RELATORIOS", "Relatórios e Dashboard"),
        ("FINANCAS", "Finanças / Taxas da plataforma"),
        ("PIX_CONFIG", "Configuração PIX"),
        ("ADMIN_USERS", "Admin — Usuários"),
        ("FISCAL_CONFIG", "Fiscal — Configurar NFC-e"),
        ("FISCAL_EMIT", "Fiscal — Emitir NFC-e"),
    ]
    for code, name in perms:
        conn.execute("INSERT OR IGNORE INTO permissions(code, name) VALUES (?,?)", (code, name))

    # Cargos padrão (idempotente)
    roles = [
        ("ADMIN", "Administrador"),
        ("VENDEDOR", "Vendedor / PDV"),
        ("GERENTE", "Gerente"),
        ("ESTOQUISTA", "Estoquista"),
        ("FINANCEIRO", "Financeiro"),
    ]
    for rc, rn in roles:
        conn.execute("INSERT OR IGNORE INTO roles(code, name) VALUES (?,?)", (rc, rn))

    # Mapear permissões padrão por cargo
    def perm_id(c): return conn.execute("SELECT id FROM permissions WHERE code=?", (c,)).fetchone()["id"]
    def role_id(c): return conn.execute("SELECT id FROM roles WHERE code=?", (c,)).fetchone()["id"]
    def link(role_code, perm_codes):
        rid = role_id(role_code)
        for pc in perm_codes:
            pid = perm_id(pc)
            conn.execute("INSERT OR IGNORE INTO role_permissions(role_id, perm_id) VALUES (?,?)", (rid, pid))

    link("ADMIN", ["PDV","ESTOQUE","RELATORIOS","FINANCAS","PIX_CONFIG","ADMIN_USERS","FISCAL_CONFIG","FISCAL_EMIT"])
    link("VENDEDOR", ["PDV","RELATORIOS"])
    link("GERENTE", ["PDV","ESTOQUE","RELATORIOS","FINANCAS"])
    link("ESTOQUISTA", ["ESTOQUE","RELATORIOS"])
    link("FINANCEIRO", ["FINANCAS","RELATORIOS"])

    # Admin padrão
    row = conn.execute("SELECT id FROM users WHERE username='admin'").fetchone()
    if not row:
        ph = make_password("admin")
        conn.execute(
            "INSERT INTO users(username, full_name, password_hash, is_admin, active, must_change_password, created_at) "
            "VALUES (?,?,?,?,?,?,?)",
            ("admin", "Administrador", ph, 1, 1, 1, now_utc_iso())
        )
        rid = role_id("ADMIN")
        uid = conn.execute("SELECT id FROM users WHERE username='admin'").fetchone()["id"]
        conn.execute("INSERT OR IGNORE INTO user_roles(user_id, role_id) VALUES (?,?)", (uid, rid))

# ---- helpers de settings
def get_setting(conn: sqlite3.Connection, key: str, default: str = "") -> str:
    r = conn.execute("SELECT value FROM settings WHERE key=?", (key,)).fetchone()
    return r["value"] if r and r["value"] is not None else default

def set_setting(conn: sqlite3.Connection, key: str, value: str):
    conn.execute(
        "INSERT INTO settings(key,value) VALUES(?,?) "
        "ON CONFLICT(key) DO UPDATE SET value=excluded.value",
        (key, value)
    )

# -----------------------------
# Usuários / permissões
# -----------------------------
def get_user_by_username(conn, username: str):
    return conn.execute("SELECT * FROM users WHERE username=?", (username,)).fetchone()

def set_user_password(conn, user_id: int, new_password: str, must_change: bool = False):
    ph = make_password(new_password)
    conn.execute("UPDATE users SET password_hash=?, must_change_password=? WHERE id=?",
                 (ph, int(bool(must_change)), user_id))

def user_has_perm(conn, user_id: int, code: str) -> bool:
    """Agora considera permissão explícita OU herdada por cargo/role."""
    # explícita
    row = conn.execute(
        "SELECT 1 FROM user_permissions up "
        "JOIN permissions p ON p.id = up.perm_id "
        "WHERE up.user_id=? AND p.code=?",
        (user_id, code)
    ).fetchone()
    if row:
        return True
    # herdada por cargo
    row2 = conn.execute(
        "SELECT 1 FROM user_roles ur "
        "JOIN role_permissions rp ON rp.role_id = ur.role_id "
        "JOIN permissions p ON p.id = rp.perm_id "
        "WHERE ur.user_id=? AND p.code=?",
        (user_id, code)
    ).fetchone()
    return bool(row2)

# ---- NOVOS helpers de roles/permissions para o Admin
def list_permissions(conn):
    return conn.execute("SELECT id, code, name FROM permissions ORDER BY name").fetchall()

def list_roles(conn):
    return conn.execute("SELECT id, code, name FROM roles ORDER BY name").fetchall()

def get_user_roles(conn, user_id: int):
    q = ("SELECT r.id, r.code, r.name FROM roles r "
         "JOIN user_roles ur ON ur.role_id=r.id WHERE ur.user_id=? ORDER BY r.name")
    return conn.execute(q, (user_id,)).fetchall()

def set_user_roles(conn, user_id: int, role_ids):
    conn.execute("DELETE FROM user_roles WHERE user_id=?", (user_id,))
    for rid in set(role_ids or []):
        conn.execute("INSERT OR IGNORE INTO user_roles(user_id, role_id) VALUES (?,?)", (user_id, rid))

def get_user_permissions(conn, user_id: int):
    q = ("SELECT p.id, p.code, p.name FROM permissions p ORDER BY p.name")
    rows = conn.execute(q).fetchall()
    role_perm_ids = set(r["id"] for r in conn.execute(
        "SELECT DISTINCT rp.perm_id FROM role_permissions rp "
        "JOIN user_roles ur ON ur.role_id = rp.role_id WHERE ur.user_id=?", (user_id,)
    ).fetchall())
    result = []
    for r in rows:
        user_checked = bool(conn.execute(
            "SELECT 1 FROM user_permissions WHERE user_id=? AND perm_id=?", (user_id, r["id"]) 
        ).fetchone())
        result.append((r["id"], r["code"], r["name"], user_checked, r["id"] in role_perm_ids))
    return result

def set_user_permissions_explicit(conn, user_id: int, perm_ids):
    conn.execute("DELETE FROM user_permissions WHERE user_id=?", (user_id,))
    for pid in set(perm_ids or []):
        conn.execute("INSERT OR IGNORE INTO user_permissions(user_id, perm_id) VALUES (?,?)", (user_id, pid))

# -----------------------------
# App
# -----------------------------
class App:
    def __init__(self, root: tk.Tk, db_path: str):
        self.root = root
        self.db_path = db_path
        self.conn = connect(self.db_path)
        init_db(self.conn)

        self.current_user = None

        # título ao lado do ícone
        self.root.title("FabricaDigitalShop — www.fabricadigitalshop.com")
        self._apply_window_icon()
        self._apply_default_geometry(1100, 700)
        self._maximize()  # <<< abre maximizado a janela principal

        self.style = ttk.Style(self.root)
        try:
            if "vista" in self.style.theme_names():
                self.style.theme_use("vista")
        except Exception:
            pass

        self._build_login()

    def _apply_window_icon(self):
        try:
            base = os.path.dirname(os.path.abspath(__file__))
        except Exception:
            base = os.getcwd()
        ico_path = os.path.join(base, "assets", "fabricadigitalshop.ico")
        png_path = os.path.join(base, "assets", "fabricadigitalshop.png")
        try:
            if os.name == "nt" and os.path.exists(ico_path):
                self.root.iconbitmap(ico_path)
            elif os.path.exists(png_path):
                icon_img = tk.PhotoImage(file=png_path)
                self.root.iconphoto(True, icon_img)
                self._icon_img_ref = icon_img
            elif os.path.exists(ico_path):
                icon_img = tk.PhotoImage(file=ico_path)
                self.root.iconphoto(True, icon_img)
                self._icon_img_ref = icon_img
        except Exception:
            pass

    def _apply_default_geometry(self, w: int, h: int):
        try:
            self.root.geometry(f"{w}x{h}")
            self.root.update_idletasks()
            sw = self.root.winfo_screenwidth()
            sh = self.root.winfo_screenheight()
            x = max((sw - w) // 2, 0)
            y = max((sh - h) // 2, 0)
            self.root.geometry(f"{w}x{h}+{x}+{y}")
        except Exception:
            self.root.geometry(f"{w}x{h}")

    def _maximize(self):
        # Tenta maximizar no Windows
        try:
            self.root.state("zoomed")
            return
        except Exception:
            pass
        # Linux / outros
        try:
            self.root.attributes("-zoomed", True)
            return
        except Exception:
            pass
        # Fallback: ocupar a tela inteira
        try:
            self.root.update_idletasks()
            sw = self.root.winfo_screenwidth()
            sh = self.root.winfo_screenheight()
            self.root.geometry(f"{sw}x{sh}+0+0")
        except Exception:
            pass

    def _maximize_window(self, win: tk.Toplevel):
        # Maximiza qualquer janela Toplevel (Admin → Usuários, etc.)
        try:
            win.state("zoomed")
            return
        except Exception:
            pass
        try:
            win.attributes("-zoomed", True)
            return
        except Exception:
            pass
        try:
            win.update_idletasks()
            sw = win.winfo_screenwidth()
            sh = win.winfo_screenheight()
            win.geometry(f"{sw}x{sh}+0+0")
        except Exception:
            pass

    # ------------- LOGIN -------------
    def _build_login(self):
        try:
            self.root.config(menu="")
        except Exception:
            pass

        self._clear_root()
        frm = ttk.Frame(self.root, padding=16)
        frm.pack(expand=True)

        title = ttk.Label(frm, text="Acesso ao Sistema", font=("Segoe UI", 18, "bold"))
        title.pack(pady=(0, 12))

        row1 = ttk.Frame(frm); row1.pack(fill="x", pady=4)
        ttk.Label(row1, text="Usuário").pack(side="left", padx=(0, 8))
        user_e = ttk.Entry(row1, width=28); user_e.pack(side="left"); user_e.insert(0, "admin")

        row2 = ttk.Frame(frm); row2.pack(fill="x", pady=4)
        ttk.Label(row2, text="Senha").pack(side="left", padx=(0, 17))
        pass_e = ttk.Entry(row2, show="•", width=28); pass_e.pack(side="left")

        def do_login(_e=None):
            u = user_e.get().strip()
            p = pass_e.get()
            if not u or not p:
                messagebox.showerror("Login", "Informe usuário e senha.")
                return
            row = get_user_by_username(self.conn, u)
            if not row or not row["active"]:
                messagebox.showerror("Login", "Usuário ou senha inválidos.")
                return
            if not check_password(p, row["password_hash"]):
                messagebox.showerror("Login", "Usuário ou senha inválidos.")
                return
            self.current_user = row
            if row["must_change_password"]:
                self._ask_change_password(row["id"])
                self.current_user = get_user_by_username(self.conn, u)
            self._build_main()

        btn = ttk.Button(frm, text="Entrar", command=do_login)
        btn.pack(pady=8)

        self.root.bind("<Return>", do_login)
        user_e.focus_set()

    def _ask_change_password(self, user_id: int):
        d = tk.Toplevel(self.root)
        d.title("Alterar Senha")
        d.transient(self.root)
        d.grab_set()
        d.resizable(False, False)
        ttk.Label(d, text="Defina uma nova senha para continuar.", font=("Segoe UI", 11)).pack(padx=16, pady=12)
        e1 = ttk.Entry(d, show="•", width=28); e1.pack(padx=16, pady=6)
        e2 = ttk.Entry(d, show="•", width=28); e2.pack(padx=16, pady=6)

        def ok():
            s1 = e1.get(); s2 = e2.get()
            if not s1 or s1 != s2:
                messagebox.showerror("Alterar Senha", "As senhas não conferem.")
                return
            set_user_password(self.conn, user_id, s1, must_change=False)
            d.destroy()
        ttk.Button(d, text="Salvar", command=ok).pack(pady=8)
        d.wait_window()

    # ------------- MAIN / HOME -------------
    def _build_main(self):
        self._clear_root()
        self._build_menubar()
        self._build_home()

    def _build_home(self):
        home = ttk.Frame(self.root, padding=20); home.pack(expand=True, fill="both")

        lbl_title = ttk.Label(home, text="FabricaDigitalShop", font=("Segoe UI", 44, "bold"))
        lbl_title.pack(pady=(50, 6))
        lbl_site = ttk.Label(home, text="www.fabricadigitalshop.com", font=("Segoe UI", 16))
        lbl_site.pack(pady=(0, 22))

        tips = ttk.Label(
            home,
            text=("""• Usuários administradores podem gerenciar acessos.
• Utilize o menu superior para abrir PDV, Estoque, Finanças e Relatórios.
• No primeiro acesso do admin, altere a senha em Conta → Trocar senha."""),  # triple-quoted, em uma única string
            anchor="center", justify="center"
        )
        tips.pack(pady=12)

        footer = ttk.Label(self.root,
                           text=f"Usuário: {self.current_user['username']} "
                                f"({'ADMIN' if self.current_user['is_admin'] else 'USER'})",
                           anchor="w")
        footer.pack(side="bottom", fill="x", padx=8, pady=6)

    def _build_menubar(self):
        menubar = tk.Menu(self.root); self.root.config(menu=menubar)

        def can(code: str) -> bool:
            return bool(self.current_user["is_admin"]) or user_has_perm(self.conn, self.current_user["id"], code)

        def _open_mod(module_name: str, func_name: str, label: str):
            try:
                m = __import__(module_name)
                fn = getattr(m, func_name, None)
                if not callable(fn):
                    raise RuntimeError(f"{module_name}.py não possui {func_name}(app, get_conn).")
                fn(self, lambda: connect(self.db_path))
            except Exception as e:
                messagebox.showerror("Módulo", f"Falha ao abrir '{label}'.\n\n{e}")

        mod = tk.Menu(menubar, tearoff=False)
        menubar.add_cascade(label="Módulos", menu=mod)
        mod.add_command(label="PDV (Vendas)", command=lambda: _open_mod("vendas", "open_pdv", "vendas"),
                        state=("normal" if can("PDV") else "disabled"))
        mod.add_command(label="Estoque", command=lambda: _open_mod("estoque", "open_estoque", "estoque"),
                        state=("normal" if can("ESTOQUE") else "disabled"))
        mod.add_command(label="Caixa (Abertura/Fechamento)", command=lambda: _open_mod("caixa", "open_caixa", "caixa"),
                        state=("normal" if can("FINANCAS") else "disabled"))
        mod.add_command(label="Finanças (Taxas)", command=lambda: _open_mod("fees", "open_fees", "fees"),
                        state=("normal" if can("FINANCAS") else "disabled"))
        mod.add_command(label="Relatórios (F11)", command=lambda: _open_mod("relatorios", "open_relatorios", "relatórios"),
                        state=("normal" if can("RELATORIOS") else "disabled"))

        fiscal = tk.Menu(menubar, tearoff=False)
        menubar.add_cascade(label="Fiscal", menu=fiscal)
        fiscal.add_command(label="Configurar Nota (NFC-e)…",
                           command=lambda: _open_mod("fiscal_config_dialog", "open_fiscal_config_dialog", "Configuração NFC-e"),
                           state=("normal" if (self.current_user["is_admin"] or can("FISCAL_CONFIG")) else "disabled"))
        fiscal.add_command(label="Configurar Nome Nota/Recibo…", command=self._open_receipt_header_dialog, state="normal")

        adm = tk.Menu(menubar, tearoff=False)
        menubar.add_cascade(label="Admin", menu=adm)
        adm.add_command(label="Usuários", command=self._admin_users,
                        state=("normal" if self.current_user["is_admin"] or can("ADMIN_USERS") else "disabled"))

        def open_pix_cfg():
            _open_mod("pix_config", "open_pix_config", "Configuração PIX")
        adm.add_command(label="Configuração PIX", command=open_pix_cfg,
                        state=("normal" if self.current_user["is_admin"] or can("PIX_CONFIG") else "disabled"))

        acct = tk.Menu(menubar, tearoff=False)
        menubar.add_cascade(label="Conta", menu=acct)
        acct.add_command(label="Trocar senha…", command=self._menu_change_password)
        acct.add_separator()
        acct.add_command(label="Sair", command=self._logout)

        # ---- NOVO: Ajuda / Manual do Usuário (F1) ----
        ajuda = tk.Menu(menubar, tearoff=False)
        menubar.add_cascade(label="Ajuda", menu=ajuda)

        def _open_manual():
            base = Path(__file__).resolve().parent
            if _open_manual_viewer:
                try:
                    _open_manual_viewer(base)
                except Exception as e:
                    messagebox.showerror("Manual", f"Não foi possível abrir o manual.\n\n{e}")
            else:
                messagebox.showinfo(
                    "Manual",
                    "Viewer do manual não encontrado.\nColoque 'tools/manual_viewer.py' no projeto e gere 'docs/manual.md'."
                )

        ajuda.add_command(label="Manual do Usuário (F1)", command=_open_manual)
        # Atalho global F1
        try:
            self.root.bind("<F1>", lambda e: _open_manual())
        except Exception:
            pass

    # --------- Diálogo: Cabeçalho de Nota/Recibo ----------
    def _open_receipt_header_dialog(self):
        w = tk.Toplevel(self.root)
        w.title("Configurar Nome Nota/Recibo")
        w.transient(self.root); w.grab_set(); w.resizable(False, False)
        frm = ttk.Frame(w, padding=12); frm.pack(fill="both", expand=True)

        with connect(self.db_path) as c:
            name = get_setting(c, "STORE_NAME", "")
            addr1 = get_setting(c, "RECEIPT_ADDR1", "")
            addr2 = get_setting(c, "RECEIPT_ADDR2", "")
            cityuf = get_setting(c, "RECEIPT_CITY_UF", "")
            doc = get_setting(c, "RECEIPT_DOC", "")

        v_name = tk.StringVar(master=frm, value=name)
        v_addr1 = tk.StringVar(master=frm, value=addr1)
        v_addr2 = tk.StringVar(master=frm, value=addr2)
        v_cityuf = tk.StringVar(master=frm, value=cityuf)
        v_doc = tk.StringVar(master=frm, value=doc)

        row = 0
        ttk.Label(frm, text="Nome do Estabelecimento *").grid(row=row, column=0, sticky="w"); row += 1
        ttk.Entry(frm, textvariable=v_name, width=46).grid(row=row, column=0, sticky="w"); row += 1
        ttk.Label(frm, text="Endereço (linha 1)").grid(row=row, column=0, sticky="w"); row += 1
        ttk.Entry(frm, textvariable=v_addr1, width=46).grid(row=row, column=0, sticky="w"); row += 1
        ttk.Label(frm, text="Endereço (linha 2)").grid(row=row, column=0, sticky="w"); row += 1
        ttk.Entry(frm, textvariable=v_addr2, width=46).grid(row=row, column=0, sticky="w"); row += 1
        ttk.Label(frm, text="Cidade/UF").grid(row=row, column=0, sticky="w"); row += 1
        ttk.Entry(frm, textvariable=v_cityuf, width=24).grid(row=row, column=0, sticky="w"); row += 1
        ttk.Label(frm, text="Documento (CNPJ/CPF) — opcional").grid(row=row, column=0, sticky="w"); row += 1
        ttk.Entry(frm, textvariable=v_doc, width=24).grid(row=row, column=0, sticky="w"); row += 1

        info = ttk.Label(frm, text="* Também usado em recibos e como fallback do PIX.", foreground="#555")
        info.grid(row=row, column=0, sticky="w", pady=(6, 0)); row += 1

        btns = ttk.Frame(frm); btns.grid(row=row, column=0, sticky="e", pady=(10,0))
        def salvar():
            nm = v_name.get().strip()
            if not nm:
                messagebox.showerror("Cabeçalho", "Informe o nome do estabelecimento.")
                return
            with connect(self.db_path) as c:
                set_setting(c, "STORE_NAME", nm)
                set_setting(c, "RECEIPT_ADDR1", v_addr1.get().strip())
                set_setting(c, "RECEIPT_ADDR2", v_addr2.get().strip())
                set_setting(c, "RECEIPT_CITY_UF", v_cityuf.get().strip())
                set_setting(c, "RECEIPT_DOC", v_doc.get().strip())
                pix_name = get_setting(c, "PIX_MERCHANT_NAME", "")
                if not pix_name:
                    set_setting(c, "PIX_MERCHANT_NAME", nm)
            messagebox.showinfo("Cabeçalho", "Dados salvos com sucesso.")
            w.destroy()

        ttk.Button(btns, text="Salvar", command=salvar).pack(side="left", padx=4)
        ttk.Button(btns, text="Fechar", command=w.destroy).pack(side="left", padx=4)

    # ------------- Admin: Usuários (CRUD + cargos + permissões) -------------
    def _admin_users(self):
        if not (self.current_user["is_admin"] or user_has_perm(self.conn, self.current_user["id"], "ADMIN_USERS")):
            messagebox.showwarning("Permissão", "Acesso negado.")
            return

        w = tk.Toplevel(self.root)
        w.title("Admin — Usuários")
        w.geometry("900x540")
        self._maximize_window(w)  # <<< abre a janela de usuários maximizada

        frm = ttk.Frame(w, padding=8); frm.pack(expand=True, fill="both")

        # Lista
        cols = ("id","username","full_name","is_admin","active")
        tree = ttk.Treeview(frm, columns=cols, show="headings", height=14)
        for c, txt in zip(cols, ("ID","Usuário","Nome","Admin","Ativo")):
            tree.heading(c, text=txt)
        tree.column("id", width=60, anchor="center")
        tree.column("username", width=160)
        tree.column("full_name", width=220)
        tree.column("is_admin", width=80, anchor="center")
        tree.column("active", width=80, anchor="center")
        tree.grid(row=0, column=0, rowspan=3, sticky="nsew")

        vsb = ttk.Scrollbar(frm, orient="vertical", command=tree.yview)
        vsb.grid(row=0, column=1, rowspan=3, sticky="ns")
        tree.configure(yscrollcommand=vsb.set)

        frm.grid_rowconfigure(0, weight=1)
        frm.grid_columnconfigure(0, weight=1)

        # Lado direito
        right = ttk.LabelFrame(frm, text="Detalhes / Permissões", padding=8)
        right.grid(row=0, column=2, sticky="nsew", padx=(10,0))
        frm.grid_columnconfigure(2, weight=1)

        v_uid = tk.IntVar(master=right, value=0)
        v_user = tk.StringVar(master=right, value="")
        v_name = tk.StringVar(master=right, value="")
        v_admin = tk.BooleanVar(master=right, value=False)
        v_active = tk.BooleanVar(master=right, value=True)

        r1 = ttk.Frame(right); r1.pack(fill="x", pady=2)
        ttk.Label(r1, text="Usuário").pack(side="left")
        e_user = ttk.Entry(r1, textvariable=v_user, width=24); e_user.pack(side="left", padx=6)
        ttk.Label(r1, text="Nome").pack(side="left", padx=(10,0))
        e_name = ttk.Entry(r1, textvariable=v_name, width=28); e_name.pack(side="left", padx=6)

        r2 = ttk.Frame(right); r2.pack(fill="x", pady=2)
        ttk.Checkbutton(r2, text="Admin", variable=v_admin).pack(side="left")
        ttk.Checkbutton(r2, text="Ativo", variable=v_active).pack(side="left", padx=10)

        # Cargos
        roles_box = ttk.LabelFrame(right, text="Cargos (roles)"); roles_box.pack(fill="x", pady=(6,4))
        roles_vars = []   # [(rid, tk.BooleanVar)]
        def load_roles_checklist():
            for child in roles_box.winfo_children(): child.destroy()
            roles_vars.clear()
            for r in list_roles(self.conn):
                rid, rcode, rname = r["id"], r["code"], r["name"]
                var = tk.BooleanVar(master=roles_box, value=False)
                chk = ttk.Checkbutton(roles_box, text=f"{rname} ({rcode})", variable=var)
                chk.pack(anchor="w")
                roles_vars.append((rid, var))

        # Permissões
        perms_box = ttk.LabelFrame(right, text="Permissões"); perms_box.pack(fill="both", expand=True, pady=(4,0))
        perms_vars = []   # [(pid, tk.BooleanVar, by_role)]
        def load_perms_checklist(user_id: int):
            for child in perms_box.winfo_children(): child.destroy()
            perms_vars.clear()
            for pid, pcode, pname, by_user, by_role in get_user_permissions(self.conn, user_id):
                var = tk.BooleanVar(master=perms_box, value=bool(by_user or by_role))
                chk = ttk.Checkbutton(perms_box, text=f"{pname} [{pcode}]", variable=var)
                chk.pack(anchor="w")
                if by_role and not by_user:
                    chk.state(["disabled"])
                perms_vars.append((pid, var, by_role))

        # Botões direita
        btns = ttk.Frame(right); btns.pack(fill="x", pady=(6,0))
        def save_user():
            uid = v_uid.get()
            u = v_user.get().strip()
            nm = v_name.get().strip()
            if not u:
                messagebox.showerror("Usuários", "Usuário obrigatório.")
                return
            if uid == 0:
                if get_user_by_username(self.conn, u):
                    messagebox.showerror("Usuários", "Usuário já existe.")
                    return
                pwd = simpledialog.askstring("Senha inicial", "Defina a senha inicial:", show="•") or "123"
                ph = make_password(pwd)
                self.conn.execute(
                    "INSERT INTO users(username, full_name, password_hash, is_admin, active, must_change_password, created_at) "
                    "VALUES (?,?,?,?,?,?,?)",
                    (u, nm, ph, int(v_admin.get()), int(v_active.get()), 1, now_utc_iso())
                )
                uid = self.conn.execute("SELECT id FROM users WHERE username=?", (u,)).fetchone()["id"]
                v_uid.set(uid)
            else:
                self.conn.execute(
                    "UPDATE users SET username=?, full_name=?, is_admin=?, active=? WHERE id=?",
                    (u, nm, int(v_admin.get()), int(v_active.get()), uid)
                )

            selected_roles = [rid for rid, var in roles_vars if var.get()]
            set_user_roles(self.conn, uid, selected_roles)

            explicit_perm_ids = [pid for pid, var, by_role in perms_vars if var.get() and not by_role]
            set_user_permissions_explicit(self.conn, uid, explicit_perm_ids)

            reload_users()
            messagebox.showinfo("Usuários", "Dados salvos.")

        def reset_pwd():
            sel = tree.selection()
            if not sel: return
            uid = int(tree.item(sel[0], "values")[0])
            newp = simpledialog.askstring("Redefinir senha", "Nova senha:", show="•")
            if not newp: return
            set_user_password(self.conn, uid, newp, must_change=False)
            messagebox.showinfo("Usuários", "Senha alterada.")

        def delete_user():
            sel = tree.selection()
            if not sel: return
            uid, uname, _, isadm, _ = tree.item(sel[0], "values")
            uid = int(uid)
            if uname == "admin":
                messagebox.showwarning("Usuários", "Não é possível remover o usuário 'admin'.")
                return
            if messagebox.askyesno("Excluir", f"Excluir o usuário '{uname}'?"):
                if isadm == "Sim":
                    n_admins = self.conn.execute(
                        "SELECT COUNT(*) FROM users WHERE is_admin=1 AND id<>?", (uid,)
                    ).fetchone()[0]
                    if n_admins == 0:
                        messagebox.showwarning("Usuários", "Não é possível excluir o último administrador.")
                        return
                self.conn.execute("DELETE FROM users WHERE id=?", (uid,))
                reload_users()
                v_uid.set(0); v_user.set(""); v_name.set(""); v_admin.set(False); v_active.set(True)
                load_roles_checklist(); load_perms_checklist(0)

        ttk.Button(btns, text="Salvar", command=save_user).pack(side="left")
        ttk.Button(btns, text="Redefinir senha", command=reset_pwd).pack(side="left", padx=6)
        ttk.Button(btns, text="Excluir", command=delete_user).pack(side="left")

        # Botões inferiores (lista)
        list_btns = ttk.Frame(frm, padding=(0,8)); list_btns.grid(row=2, column=0, sticky="w")
        def add_new():
            v_uid.set(0); v_user.set(""); v_name.set(""); v_admin.set(False); v_active.set(True)
            load_roles_checklist(); load_perms_checklist(0); e_user.focus_set()
        ttk.Button(list_btns, text="Novo", command=add_new).pack(side="left")

        def toggle_active():
            sel = tree.selection()
            if not sel: return
            uid = int(tree.item(sel[0], "values")[0])
            r = self.conn.execute("SELECT active FROM users WHERE id=?", (uid,)).fetchone()
            self.conn.execute("UPDATE users SET active=? WHERE id=?", (0 if r["active"] else 1, uid))
            reload_users()
        ttk.Button(list_btns, text="Ativar/Desativar (rápido)", command=toggle_active).pack(side="left", padx=6)

        # Carregamento/seleção
        def reload_users():
            tree.delete(*tree.get_children())
            for r in self.conn.execute("SELECT id,username,full_name,is_admin,active FROM users ORDER BY id"):
                tree.insert("", "end", values=(r["id"], r["username"], r["full_name"],
                                               "Sim" if r["is_admin"] else "Não",
                                               "Sim" if r["active"] else "Não"))

        def on_select(_e=None):
            sel = tree.selection()
            if not sel: return
            uid, uname, fname, isadm, active = tree.item(sel[0], "values")
            uid = int(uid)
            v_uid.set(uid); v_user.set(uname); v_name.set(fname or "")
            v_admin.set(isadm == "Sim"); v_active.set(active == "Sim")

            load_roles_checklist()
            current_roles = {r["id"] for r in get_user_roles(self.conn, uid)}
            for rid, var in roles_vars:
                var.set(rid in current_roles)

            load_perms_checklist(uid)

        tree.bind("<<TreeviewSelect>>", on_select)

        reload_users()
        load_roles_checklist()
        load_perms_checklist(0)

    # ------------- Conta -------------
    def _menu_change_password(self):
        if not self.current_user:
            return
        u = self.current_user
        d = tk.Toplevel(self.root)
        d.title("Trocar Senha")
        d.transient(self.root); d.grab_set()
        ttk.Label(d, text=f"Usuário: {u['username']}").pack(padx=12, pady=(12,6))
        e1 = ttk.Entry(d, show="•", width=28); e1.pack(padx=12, pady=4)
        e2 = ttk.Entry(d, show="•", width=28); e2.pack(padx=12, pady=4)
        def salvar():
            s1, s2 = e1.get(), e2.get()
            if not s1 or s1 != s2:
                messagebox.showerror("Conta", "As senhas não conferem.")
                return
            set_user_password(self.conn, u["id"], s1, must_change=False)
            messagebox.showinfo("Conta", "Senha alterada com sucesso.")
            d.destroy()
        ttk.Button(d, text="Salvar", command=salvar).pack(pady=8)
        d.wait_window()

    def _logout(self):
        if messagebox.askyesno("Sair", "Deseja encerrar a sessão?"):
            self.current_user = None
            self._build_login()

    # ------------- Utils -------------
    def _clear_root(self):
        for w in self.root.winfo_children():
            w.destroy()

# -----------------------------
# CLI util: reset admin
# -----------------------------
def cli_reset_admin(db_path: str):
    with closing(connect(db_path)) as conn:
        init_db(conn)
        row = conn.execute("SELECT id FROM users WHERE username='admin'").fetchone()
        if row:
            set_user_password(conn, row["id"], "admin", must_change=True)
            conn.execute("UPDATE users SET is_admin=1, active=1 WHERE id=?", (row["id"],))
            print("Senha do admin redefinida para 'admin'. Será solicitada a troca no próximo login.")
        else:
            ph = make_password("admin")
            conn.execute(
                "INSERT INTO users(username, full_name, password_hash, is_admin, active, must_change_password, created_at) "
                "VALUES (?,?,?,?,?,?,?)",
                ("admin", "Administrador", ph, 1, 1, 1, now_utc_iso())
            )
            print("Usuário 'admin' criado com senha 'admin' (será solicitada a troca).")

# -----------------------------
# Main
# -----------------------------
def main(argv=None):
    argv = argv or sys.argv[1:]
    db_path = get_db_path()

    if "--reset-admin" in argv:
        cli_reset_admin(db_path)
        return 0

    root = tk.Tk()
    app = App(root, db_path)
    try:
        root.mainloop()
    except KeyboardInterrupt:
        pass
    return 0

if __name__ == "__main__":
    raise SystemExit(main())
