# estoque.py
from __future__ import annotations

import sqlite3
import tkinter as tk
from tkinter import ttk, messagebox, simpledialog
from decimal import Decimal, ROUND_HALF_UP
from typing import Callable, Optional, Tuple

# --- dependências opcionais para webcam ---
try:
    import cv2  # opencv (QR) + opencv-contrib (barcodes 1D)
except Exception:
    cv2 = None
try:
    from PIL import Image, ImageTk
except Exception:
    Image = ImageTk = None
try:
    from pyzbar import pyzbar  # fallback robusto p/ 1D e QR
except Exception:
    pyzbar = None


# ----------------------------- schema / migrações defensivas -----------------------------
EXTRA_SCHEMA_SQL = """
CREATE TABLE IF NOT EXISTS products (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    sku TEXT UNIQUE NOT NULL,
    name TEXT NOT NULL,
    unit_price REAL NOT NULL,
    stock_qty INTEGER NOT NULL DEFAULT 0
);
CREATE INDEX IF NOT EXISTS ix_products_sku ON products(sku);
"""

def _ensure_schema(get_conn: Callable) -> None:
    def _has_column(conn: sqlite3.Connection, table: str, col: str) -> bool:
        cur = conn.execute(f'PRAGMA table_info("{table}")')
        return any(r[1].lower() == col.lower() for r in cur.fetchall())

    def _add_column(conn: sqlite3.Connection, table: str, col: str, ddl_tail: str) -> None:
        conn.execute(f'ALTER TABLE "{table}" ADD COLUMN "{col}" {ddl_tail}')

    with get_conn() as conn:
        conn.executescript(EXTRA_SCHEMA_SQL)
        if not _has_column(conn, "products", "stock_qty"):
            _add_column(conn, "products", "stock_qty", "INTEGER NOT NULL DEFAULT 0")


# ----------------------------- util -----------------------------
def _money(x) -> Decimal:
    d = x if isinstance(x, Decimal) else Decimal(str(x))
    return d.quantize(Decimal("0.01"), rounding=ROUND_HALF_UP)

def _fmt_brl(d: Decimal) -> str:
    s = f"{d:.2f}"
    return s.replace(".", ",")

def _parse_decimal(s: str) -> Decimal:
    if s is None or s.strip() == "":
        return Decimal("0")
    s = s.strip().replace(".", "").replace(",", ".")
    try:
        return _money(Decimal(s))
    except Exception:
        return Decimal("0.00")


# ============================= janela principal: Estoque =============================
class EstoqueWindow:
    def __init__(self, app, get_conn: Callable):
        self.app = app
        self.get_conn = get_conn
        _ensure_schema(self.get_conn)

        self.win = tk.Toplevel(app.root)
        self.win.title("Estoque — Produtos")
        self.win.geometry("1000x660")

        self.var_q = tk.StringVar()

        # toolbar/topo
        top = ttk.Frame(self.win, padding=10)
        top.pack(fill="x")
        ttk.Label(top, text="Buscar (SKU ou nome):").pack(side="left")
        ent = ttk.Entry(top, textvariable=self.var_q, width=40)
        ent.pack(side="left", padx=6)
        ent.focus_set()
        ttk.Button(top, text="Pesquisar (Enter)", command=self.refresh).pack(side="left", padx=(0, 6))
        ttk.Button(top, text="Ler QR/CB (F7)", command=self.scan_to_filter).pack(side="left")
        self.win.bind("<Return>", lambda e: self.refresh())
        self.win.bind("<F7>", lambda e: self.scan_to_filter())

        # lista
        cols = ("sku", "name", "price", "stock")
        self.tv = ttk.Treeview(self.win, columns=cols, show="headings", height=22)
        for c, t, w, a in [
            ("sku", "SKU", 160, "w"),
            ("name", "Produto", 520, "w"),
            ("price", "Preço", 110, "e"),
            ("stock", "Estoque", 90, "e"),
        ]:
            self.tv.heading(c, text=t)
            self.tv.column(c, width=w, anchor=a)
        self.tv.pack(fill="both", expand=True, padx=10, pady=(0, 10))
        self.tv.bind("<Double-1>", lambda e: self.edit_product())

        # ações
        actions = ttk.Frame(self.win, padding=10)
        actions.pack(fill="x")
        ttk.Button(actions, text="Novo (Ctrl+N)", command=self.new_product).pack(side="left")
        ttk.Button(actions, text="Editar (Enter ou Ctrl+E)", command=self.edit_product).pack(side="left", padx=6)
        ttk.Button(actions, text="Excluir (Del)", command=self.delete_product).pack(side="left", padx=6)
        ttk.Button(actions, text="Ajustar estoque (Ctrl+J)", command=self.adjust_stock).pack(side="left", padx=6)
        ttk.Button(actions, text="Atualizar (F5)", command=self.refresh).pack(side="left", padx=6)

        self.win.bind_all("<Control-n>", lambda e: self.new_product())
        self.win.bind_all("<Control-e>", lambda e: self.edit_product())
        self.win.bind("<Delete>", lambda e: self.delete_product())
        self.win.bind_all("<Control-j>", lambda e: self.adjust_stock())
        self.win.bind("<F5>", lambda e: self.refresh())

        self.refresh()

    # ---------- CRUD ----------
    def refresh(self):
        q = self.var_q.get().strip()
        like = f"%{q}%"
        with self.get_conn() as conn:
            conn.row_factory = sqlite3.Row
            if q:
                cur = conn.execute(
                    "SELECT id, sku, name, unit_price, stock_qty FROM products "
                    "WHERE sku LIKE ? COLLATE NOCASE OR name LIKE ? COLLATE NOCASE "
                    "ORDER BY name",
                    (like, like),
                )
            else:
                cur = conn.execute(
                    "SELECT id, sku, name, unit_price, stock_qty FROM products ORDER BY name"
                )
            rows = cur.fetchall()

        for i in self.tv.get_children():
            self.tv.delete(i)
        for r in rows:
            self.tv.insert("", "end", iid=str(r["id"]), values=(
                r["sku"], r["name"], _fmt_brl(_money(r["unit_price"])), r["stock_qty"]
            ))

    def _selected_id(self) -> Optional[int]:
        sel = self.tv.selection()
        if not sel:
            return None
        try:
            return int(sel[0])
        except Exception:
            return None

    def new_product(self):
        ProductForm(self, self.get_conn, title="Novo produto")

    def edit_product(self):
        pid = self._selected_id()
        if pid is None:
            messagebox.showinfo("Editar", "Selecione um produto.")
            return
        ProductForm(self, self.get_conn, title="Editar produto", product_id=pid)

    def delete_product(self):
        pid = self._selected_id()
        if pid is None:
            messagebox.showinfo("Excluir", "Selecione um produto.")
            return
        if not messagebox.askyesno("Confirmar", "Excluir este produto?"):
            return
        with self.get_conn() as conn:
            # impede excluir se já houve venda
            used = conn.execute(
                "SELECT COUNT(*) FROM sale_items WHERE product_id=?", (pid,)
            ).fetchone()[0]
            if used:
                messagebox.showwarning(
                    "Não permitido",
                    "Este produto já foi utilizado em vendas e não pode ser excluído."
                )
                return
            conn.execute("DELETE FROM products WHERE id=?", (pid,))
        self.refresh()

    def adjust_stock(self):
        pid = self._selected_id()
        if pid is None:
            messagebox.showinfo("Ajuste de estoque", "Selecione um produto.")
            return
        qty_str = simpledialog.askstring(
            "Ajustar estoque",
            "Informe o ajuste (ex.: 10 para somar, -5 para subtrair):",
            parent=self.win,
        )
        if qty_str is None:
            return
        try:
            qty = int(qty_str.strip())
        except Exception:
            messagebox.showerror("Inválido", "Quantidade inválida.")
            return
        with self.get_conn() as conn:
            cur = conn.execute("SELECT stock_qty FROM products WHERE id=?", (pid,))
            row = cur.fetchone()
            if not row:
                return
            new_q = row[0] + qty
            if new_q < 0:
                messagebox.showerror("Inválido", "Resultado negativo de estoque.")
                return
            conn.execute("UPDATE products SET stock_qty=? WHERE id=?", (new_q, pid))
        self.refresh()

    # ---------- scanner na tela principal ----------
    def scan_to_filter(self):
        if cv2 is None or ImageTk is None:
            messagebox.showerror(
                "Scanner indisponível",
                "Para usar a câmera, instale:\n\n"
                "  pip install pillow opencv-contrib-python pyzbar"
            )
            return
        ScannerWindow(self.win, on_code=self._scanner_filter_cb)

    def _scanner_filter_cb(self, text: str) -> bool:
        code = (text or "").strip()
        if not code:
            return False
        self.var_q.set(code)
        # tenta localizar por SKU exato
        with self.get_conn() as conn:
            row = conn.execute(
                "SELECT id FROM products WHERE LOWER(sku)=LOWER(?)", (code,)
            ).fetchone()
        if row:
            self.refresh()
            # seleciona o item encontrado
            iid = str(row[0])
            if iid in self.tv.get_children():
                self.tv.selection_set(iid)
                self.tv.see(iid)
            return True  # fecha scanner
        else:
            self.refresh()
            # oferece cadastrar
            if messagebox.askyesno("Produto não cadastrado",
                                   f"Código lido: {code}\n\nDeseja cadastrar agora?"):
                ProductForm(self, self.get_conn, title="Novo produto", preset_sku=code)
                return True
            return False  # mantém scanner aberto


# ============================= formulário de produto =============================
class ProductForm:
    def __init__(
        self,
        parent: EstoqueWindow,
        get_conn: Callable,
        title: str,
        product_id: Optional[int] = None,
        preset_sku: Optional[str] = None,
    ):
        self.parent = parent
        self.get_conn = get_conn
        self.product_id = product_id

        self.win = tk.Toplevel(parent.win)
        self.win.title(title)
        self.win.geometry("560x300")
        self.win.transient(parent.win)
        self.win.grab_set()

        self.var_sku = tk.StringVar(value=preset_sku or "")
        self.var_name = tk.StringVar(value="")
        self.var_price = tk.StringVar(value="0,00")
        self.var_stock = tk.StringVar(value="0")

        form = ttk.Frame(self.win, padding=10)
        form.pack(fill="both", expand=True)

        # SKU
        ttk.Label(form, text="SKU / Código de barras:").grid(row=0, column=0, sticky="w")
        e_sku = ttk.Entry(form, textvariable=self.var_sku, width=28)
        e_sku.grid(row=0, column=1, sticky="w", padx=(6, 6))
        ttk.Button(form, text="Ler QR/CB (F7)", command=self.scan_to_sku).grid(row=0, column=2, sticky="w")
        self.win.bind("<F7>", lambda e: self.scan_to_sku())

        # Nome
        ttk.Label(form, text="Nome:").grid(row=1, column=0, sticky="w", pady=(10, 0))
        ttk.Entry(form, textvariable=self.var_name, width=48).grid(row=1, column=1, columnspan=2, sticky="w", padx=(6, 0), pady=(10, 0))

        # Preço
        ttk.Label(form, text="Preço unitário (R$):").grid(row=2, column=0, sticky="w", pady=(10, 0))
        ttk.Entry(form, textvariable=self.var_price, width=18).grid(row=2, column=1, sticky="w", padx=(6, 0), pady=(10, 0))

        # Estoque
        ttk.Label(form, text="Estoque inicial:").grid(row=3, column=0, sticky="w", pady=(10, 0))
        ttk.Entry(form, textvariable=self.var_stock, width=10).grid(row=3, column=1, sticky="w", padx=(6, 0), pady=(10, 0))

        # botões
        btns = ttk.Frame(form)
        btns.grid(row=4, column=0, columnspan=3, sticky="w", pady=(18, 0))
        ttk.Button(btns, text="Salvar (Ctrl+S)", command=self.save).pack(side="left")
        ttk.Button(btns, text="Cancelar (Esc)", command=self.win.destroy).pack(side="left", padx=8)
        self.win.bind_all("<Control-s>", lambda e: self.save())
        self.win.bind("<Escape>", lambda e: self.win.destroy())

        # se edição, carrega dados
        if self.product_id:
            with self.get_conn() as conn:
                conn.row_factory = sqlite3.Row
                r = conn.execute(
                    "SELECT sku, name, unit_price, stock_qty FROM products WHERE id=?",
                    (self.product_id,),
                ).fetchone()
            if r:
                self.var_sku.set(r["sku"])
                self.var_name.set(r["name"])
                self.var_price.set(_fmt_brl(_money(r["unit_price"])))
                self.var_stock.set(str(r["stock_qty"]))

    def scan_to_sku(self):
        if cv2 is None or ImageTk is None:
            messagebox.showerror(
                "Scanner indisponível",
                "Para usar a câmera, instale:\n\n"
                "  pip install pillow opencv-contrib-python pyzbar"
            )
            return
        ScannerWindow(self.win, on_code=self._on_scanned)

    def _on_scanned(self, txt: str) -> bool:
        t = (txt or "").strip()
        if not t:
            return False
        self.var_sku.set(t)
        return True  # fecha scanner

    def save(self):
        # valida
        sku = self.var_sku.get().strip()
        name = self.var_name.get().strip()
        try:
            price = _parse_decimal(self.var_price.get())
            if price <= 0:
                raise ValueError
        except Exception:
            messagebox.showerror("Inválido", "Preço inválido.")
            return
        try:
            stock = int(self.var_stock.get().strip())
            if stock < 0:
                raise ValueError
        except Exception:
            messagebox.showerror("Inválido", "Estoque inválido.")
            return
        if not sku or not name:
            messagebox.showerror("Inválido", "Preencha SKU e Nome.")
            return

        with self.get_conn() as conn:
            # checa SKU duplicado
            if self.product_id:
                row = conn.execute(
                    "SELECT COUNT(*) FROM products WHERE LOWER(sku)=LOWER(?) AND id<>?",
                    (sku, self.product_id),
                ).fetchone()
            else:
                row = conn.execute(
                    "SELECT COUNT(*) FROM products WHERE LOWER(sku)=LOWER(?)",
                    (sku,),
                ).fetchone()
            if row[0]:
                messagebox.showerror("Duplicado", "Já existe um produto com esse SKU.")
                return

            if self.product_id:
                conn.execute(
                    "UPDATE products SET sku=?, name=?, unit_price=?, stock_qty=? WHERE id=?",
                    (sku, name, float(_money(price)), stock, self.product_id),
                )
            else:
                conn.execute(
                    "INSERT INTO products (sku, name, unit_price, stock_qty) VALUES (?,?,?,?)",
                    (sku, name, float(_money(price)), stock),
                )
        self.parent.refresh()
        self.win.destroy()


# ============================= janela de scanner (reutilizável) =============================
class ScannerWindow:
    """
    Webcam scanner:
      - QR via cv2.QRCodeDetector
      - Códigos de barras 1D via BarcodeDetector (opencv-contrib) e/ou pyzbar (fallback)
    on_code(text) -> bool: True fecha; False continua.
    """
    def __init__(self, parent: tk.Tk, on_code):
        self.on_code = on_code
        self.win = tk.Toplevel(parent)
        self.win.title("Scanner — QR / Código de Barras")
        self.win.geometry("820x560")
        self.win.transient(parent)
        self.win.grab_set()
        self.running = True
        self._last_sent = None

        top = ttk.Frame(self.win, padding=6)
        top.pack(fill="x")
        self.lbl_info = ttk.Label(top, text="Abrindo câmera…")
        self.lbl_info.pack(side="left")
        ttk.Button(top, text="Fechar (Esc)", command=self.close).pack(side="right")
        self.win.bind("<Escape>", lambda e: self.close())

        self.preview = ttk.Label(self.win)
        self.preview.pack(fill="both", expand=True, padx=8, pady=8)

        # open camera
        self.cap = cv2.VideoCapture(0, cv2.CAP_DSHOW) if cv2 is not None else None
        if not self.cap or not self.cap.isOpened():
            messagebox.showerror("Câmera", "Não foi possível abrir a webcam.")
            self.close()
            return
        self.cap.set(cv2.CAP_PROP_FRAME_WIDTH, 1280)
        self.cap.set(cv2.CAP_PROP_FRAME_HEIGHT, 720)

        self.qr = cv2.QRCodeDetector() if cv2 is not None else None
        self.bar = None
        mode = ["QR"]
        if cv2 is not None:
            try:
                bd = getattr(cv2, "barcode_BarcodeDetector", None)
                if bd:
                    self.bar = bd()
                else:
                    mod = getattr(cv2, "barcode", None)
                    if mod and hasattr(mod, "BarcodeDetector"):
                        self.bar = mod.BarcodeDetector()
                if self.bar:
                    mode.append("Barras(OpenCV)")
            except Exception:
                self.bar = None
        if pyzbar is not None:
            mode.append("Barras(pyzbar)")
        self.lbl_info.config(text="Modo: " + " + ".join(mode) + " — Esc para sair")

        self._loop()

    def _loop(self):
        if not self.running:
            return
        ok, frame = self.cap.read()
        if not ok:
            self.win.after(30, self._loop)
            return

        decoded_texts = []
        boxes = []

        # QR
        try:
            if self.qr is not None:
                if hasattr(self.qr, "detectAndDecodeMulti"):
                    texts, pts, _ = self.qr.detectAndDecodeMulti(frame)
                    for t in texts or []:
                        if t:
                            decoded_texts.append(t)
                    if pts is not None:
                        for p in pts:
                            boxes.append(p)
                else:
                    t, pts = self.qr.detectAndDecode(frame)
                    if t:
                        decoded_texts.append(t)
                    if pts is not None and len(pts) > 0:
                        boxes.append(pts)
        except Exception:
            pass

        # Barras (OpenCV contrib)
        try:
            if self.bar is not None:
                okb, infos, types, bpts = self.bar.detectAndDecode(frame)
                if okb and infos:
                    for t in infos:
                        if t:
                            decoded_texts.append(t)
                if bpts is not None and len(bpts) > 0:
                    for p in bpts:
                        boxes.append(p)
        except Exception:
            pass

        # Fallback pyzbar
        try:
            if pyzbar is not None:
                gray = cv2.cvtColor(frame, cv2.COLOR_BGR2GRAY)
                for rot in [0, 1, 2, 3]:
                    g = gray
                    if rot:
                        g = cv2.rotate(g, [cv2.ROTATE_90_CLOCKWISE, cv2.ROTATE_180, cv2.ROTATE_90_COUNTERCLOCKWISE][rot-1])
                    decs = pyzbar.decode(g)
                    if decs:
                        for d in decs:
                            txt = d.data.decode("utf-8", errors="ignore").strip()
                            if txt:
                                decoded_texts.append(txt)
                            pts = [(point.x, point.y) for point in d.polygon] or []
                            if pts:
                                boxes.append(pts)
                        break
        except Exception:
            pass

        # desenha caixas
        try:
            for p in boxes:
                pts = []
                if hasattr(p, "astype"):  # numpy
                    pts = p.astype(int).reshape(-1, 2).tolist()
                else:
                    pts = [(int(x), int(y)) for (x, y) in p]
                for i in range(len(pts)):
                    cv2.line(frame, pts[i], pts[(i + 1) % len(pts)], (0, 255, 0), 2)
        except Exception:
            pass

        # escolhe primeiro texto novo
        found_text = None
        for t in decoded_texts:
            t = str(t).strip()
            if t and t != self._last_sent:
                found_text = t
                break

        # render no Tk
        if ImageTk is not None:
            rgb = cv2.cvtColor(frame, cv2.COLOR_BGR2RGB)
            im = Image.fromarray(rgb)
            im = im.resize((800, int(800 * rgb.shape[0] / rgb.shape[1])), Image.BILINEAR)
            tkimg = ImageTk.PhotoImage(image=im)
            self.preview.configure(image=tkimg)
            self.preview.image = tkimg

        if found_text:
            self._last_sent = found_text
            try:
                should_close = bool(self.on_code(found_text))
            except Exception:
                should_close = False
            if should_close:
                self.close()
                return

        self.win.after(20, self._loop)

    def close(self):
        self.running = False
        try:
            if self.cap is not None:
                self.cap.release()
        except Exception:
            pass
        try:
            self.win.destroy()
        except Exception:
            pass


# ============================= API pública =============================
def open_estoque(app, get_conn: Callable):
    """Abre a janela de estoque. Usa app.root e a função get_conn do seu projeto."""
    return EstoqueWindow(app, get_conn)
