# fee_gateway.py
from __future__ import annotations
import os, json
from typing import Optional, Tuple

try:
    import requests
    _REQ = True
except Exception:
    _REQ = False

# ====== CREDENCIAIS DO APP (embutidas) ======
# ATENÇÃO: Substituir pelos seus dados de produção/sandbox do APP (não do cliente!)
_APP_MP_TOKEN = "APP_USR_SEU_ACCESS_TOKEN_DO_APP_AQUI"
_APP_MP_BASE  = "https://api.mercadopago.com"
_APP_PAYER_EMAIL = "licenca@fabricadigital.shop"   # e-mail do pagador exigido pela API
_APP_DESC = "Licença FabricaDigital.shop (taxas do app)"

# ====== Atualização remota assinada (opcional) ======
_REMOTE_CONF_URL = None  # ex: "https://seu-cdn/fee-gw.json.sig"
_PUBLIC_KEY_PEM  = None  # PEM string de chave pública para validar assinatura
_HAS_CRYPTO = False
try:
    from cryptography.hazmat.primitives import serialization, hashes
    from cryptography.hazmat.primitives.asymmetric import padding
    _HAS_CRYPTO = True
except Exception:
    _HAS_CRYPTO = False

_cached_token = _APP_MP_TOKEN
_cached_base  = _APP_MP_BASE

def refresh_remote_config() -> None:
    """Opcional: baixa config assinada. Se falhar, mantém embutida."""
    global _cached_token, _cached_base
    if not (_REQ and _REMOTE_CONF_URL and _HAS_CRYPTO and _PUBLIC_KEY_PEM):
        return
    try:
        r = requests.get(_REMOTE_CONF_URL, timeout=6)
        if r.status_code != 200: return
        # Formato esperado: {"payload": "...json...", "signature": "base64"}
        data = r.json()
        payload = data.get("payload","").encode("utf-8")
        signature_b64 = data.get("signature","")
        import base64
        sig = base64.b64decode(signature_b64)

        pub = serialization.load_pem_public_key(_PUBLIC_KEY_PEM.encode("utf-8"))
        pub.verify(sig, payload, padding.PKCS1v15(), hashes.SHA256())

        conf = json.loads(payload.decode("utf-8"))
        tok = conf.get("mp_token") or _cached_token
        bas = conf.get("api_base") or _cached_base
        if isinstance(tok,str) and tok and isinstance(bas,str) and bas.startswith("http"):
            _cached_token = tok
            _cached_base  = bas
    except Exception:
        # Segurança: se algo falhar, ignora atualização.
        return

def create_pix_charge(total_amount: float) -> Optional[Tuple[str, str]]:
    """
    Cria cobrança Pix no Mercado Pago com credenciais do APP.
    Retorna: (qr_code, payment_id) ou None em falha.
    """
    if not _REQ: return None
    refresh_remote_config()
    url = f"{_cached_base.rstrip('/')}/v1/payments"
    headers = {
        "Authorization": f"Bearer {_cached_token}",
        "Content-Type": "application/json",
    }
    # idempotência simples: usar um sufixo pseudo-aleatório
    import random, string
    idem = "FEE" + "".join(random.choice(string.ascii_uppercase+string.digits) for _ in range(10))
    headers["X-Idempotency-Key"] = idem

    body = {
        "transaction_amount": float(total_amount),
        "description": _APP_DESC,
        "payment_method_id": "pix",
        "external_reference": idem,
        "payer": {"email": _APP_PAYER_EMAIL},
    }
    try:
        r = requests.post(url, json=body, headers=headers, timeout=15)
        if r.status_code >= 400:
            return None
        data = r.json()
        poi = data.get("point_of_interaction", {}).get("transaction_data", {})
        payload = poi.get("qr_code")
        payment_id = str(data.get("id"))
        if payload and payment_id:
            return payload, payment_id
    except Exception:
        pass
    return None

def check_paid(payment_id: str) -> bool:
    """Consulta status no MP (APP). True se 'approved'."""
    if not (_REQ and payment_id): return False
    refresh_remote_config()
    url = f"{_cached_base.rstrip('/')}/v1/payments/{payment_id}"
    headers = {"Authorization": f"Bearer {_cached_token}"}
    try:
        r = requests.get(url, headers=headers, timeout=10)
        if r.status_code >= 400:
            return False
        data = r.json()
        return (data.get("status","").lower() == "approved")
    except Exception:
        return False
