# fees.py — mostra pendências do dia e de todas as datas; cobra tudo e desbloqueia 
# Rev: 2025-10-17 r3

from __future__ import annotations

import hmac
import hashlib
import random
import sqlite3
import string
from datetime import datetime, timedelta, time as dtime, timezone
from typing import Callable, List, Optional, Tuple

import tkinter as tk
from tkinter import filedialog, messagebox, ttk

__REV__ = "fees.py UI+totais 2025-10-17 r3"

# ============================ Config ============================

PLATFORM_PIX_KEY = "a1ecbd69-0cd7-4155-b2f2-f4a6a0977421"
MP_ACCESS_TOKEN = (
    "APP_USR-8822916334700678-082909-1bc44aa9459bd9ef0eafb0228692abbe-94067540"
)

MP_API_BASE = "https://api.mercadopago.com"
MP_TIMEOUT = 15
POLL_INTERVAL_S = 3
MP_MIN_AMOUNT = 1.00  # mínimo do provedor

CUTOFF_FIXED_STR = "00:00 (fixo, hora de rede)"
OFFLINE_GRACE_HOURS = 24
SAO_PAULO_TZ = timezone(timedelta(hours=-3))

# ============================ Deps opcionais ============================

try:
    import requests

    _REQ_OK = True
except Exception:
    _REQ_OK = False

try:
    import qrcode
    from PIL import Image, ImageTk

    _QR_OK = True
except Exception:
    _QR_OK = False
    Image = ImageTk = None  # type: ignore

# ============================ Util: moeda ============================


def brl(v: float) -> str:
    """Formata valor no padrão BRL sem depender de locale."""
    try:
        return ("R$ " + format(float(v or 0.0), ",.2f")).replace(",", "X").replace(".", ",").replace("X", ".")
    except Exception:
        return "R$ 0,00"


# ============================ Selo HMAC ============================

# >>>>>> Use a mesma CHAVE dos outros módulos relacionados <<<<<<
_HMAC_SECRET = b"\xcc\x8cW\xa1R\xe3r|3qFt\x8e`\xd6v\t\x0f\x12\x82\xef\x1c\xb1\xf0\x0b\xb6\xd5a\x01\xa6\xa0;"

_SEAL_FIELDS = [
    "id",
    "created_at",
    "amount",
    "status",
    "sale_id",
    "payment_id",
    "paid_at",
    "provider",
    "provider_status",
]


def _seal_from_row_dict(d: dict) -> str:
    def _n(v):
        if v is None:
            return ""
        s = str(v).strip()
        return s.replace("\r", "").replace("\n", " ")

    payload = "|".join(_n(d.get(k)) for k in _SEAL_FIELDS).encode("utf-8")
    return hmac.new(_HMAC_SECRET, payload, hashlib.sha256).hexdigest()


def _reseal_row(cx: sqlite3.Connection, table: str, row_id: int) -> None:
    try:
        r = cx.execute(
            f"""
            SELECT id, COALESCE(created_at,'') created_at, amount, status, sale_id,
                   payment_id, paid_at, provider, provider_status
              FROM {table} WHERE id=?
            """,
            (row_id,),
        ).fetchone()
        if not r:
            return
        # sqlite3.Row → map-like quando row_factory configurada; protegemos aqui:
        try:
            d = {k: r[k] for k in r.keys()}  # type: ignore[attr-defined]
        except Exception:
            d = {}
        seal = _seal_from_row_dict(d)
        cx.execute(f"UPDATE {table} SET seal=? WHERE id=?", (seal, row_id))
    except Exception:
        pass


# ============================ DB utils ============================


def _has_table(cx: sqlite3.Connection, name: str) -> bool:
    r = cx.execute(
        "SELECT 1 FROM sqlite_master WHERE (type='table' OR type='view') AND lower(name)=lower(?)",
        (name,),
    ).fetchone()
    return bool(r)


def _cols(cx: sqlite3.Connection, table: str) -> List[str]:
    try:
        return [r[1].lower() for r in cx.execute(f'PRAGMA table_info("{table}")')]
    except Exception:
        return []


def _table_info_dict(cx: sqlite3.Connection, table: str) -> dict:
    info = {}
    for cid, name, ctype, notnull, dflt_value, pk in cx.execute(f'PRAGMA table_info("{table}")'):
        info[name.lower()] = (int(notnull or 0), dflt_value)
    return info


def _ensure_min_tables(cx: sqlite3.Connection) -> None:
    cx.execute(
        """
        CREATE TABLE IF NOT EXISTS settings(
            key TEXT PRIMARY KEY,
            value TEXT
        )
        """
    )

    cx.execute(
        """
        CREATE TABLE IF NOT EXISTS taxas_cobrancas(
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            created_at TEXT NOT NULL,
            status TEXT NOT NULL DEFAULT 'PENDENTE',
            amount REAL,
            sale_id INTEGER,
            payment_id TEXT
        )
        """
    )

    tcols = _cols(cx, "taxas_cobrancas")
    if "valor" in tcols and "amount" not in tcols:
        cx.execute("ALTER TABLE taxas_cobrancas ADD COLUMN amount REAL")
        cx.execute("UPDATE taxas_cobrancas SET amount=COALESCE(valor,0.10)")
    elif "valor" in tcols and "amount" in tcols:
        cx.execute("UPDATE taxas_cobrancas SET amount=COALESCE(amount, valor)")
        cx.execute("UPDATE taxas_cobrancas SET valor = COALESCE(valor, amount)")

    for newcol in ("paid_at", "provider", "provider_status", "provider_payload_snippet", "seal"):
        if newcol not in tcols:
            try:
                cx.execute(f'ALTER TABLE taxas_cobrancas ADD COLUMN {newcol} TEXT')
            except Exception:
                pass

    info = list(cx.execute('PRAGMA table_info("taxas_cobrancas")'))
    flags = {r[1].lower(): r[3] for r in info}
    for extra in ("txid", "copia_cola"):
        if flags.get(extra, 0) == 1:
            cx.execute(f"UPDATE taxas_cobrancas SET {extra}='' WHERE {extra} IS NULL")


def _today_local() -> str:
    return datetime.now().strftime("%Y-%m-%d")


def _existing_fee_tables(cx: sqlite3.Connection) -> List[str]:
    tables: List[str] = []
    if _has_table(cx, "taxas_cobrancas"):
        tables.append("taxas_cobrancas")
    if _has_table(cx, "platform_fees"):
        tables.append("platform_fees")
    return tables


_FEE_DATE_COL_CANDIDATES = ("created_at", "created", "datetime", "ts", "data", "data_hora", "timestamp")


def _fee_date_col(cx: sqlite3.Connection, table: str) -> Optional[str]:
    cset = set(_cols(cx, table))
    for c in _FEE_DATE_COL_CANDIDATES:
        if c in cset:
            return c
    return None


def _sum_pending_today(cx: sqlite3.Connection) -> Tuple[int, float]:
    """Conta pendências do DIA (status='PENDENTE')."""
    tables = _existing_fee_tables(cx)
    if not tables:
        return 0, 0.0

    total_qtd = 0
    total_val = 0.0
    today = _today_local()

    for table in tables:
        date_col = _fee_date_col(cx, table)
        if not date_col:
            continue

        def _try(col: str) -> Tuple[int, float]:
            row = cx.execute(
                f"""
                SELECT COUNT(*), COALESCE(SUM(COALESCE({col},0.10)),0)
                  FROM {table}
                 WHERE UPPER(status)='PENDENTE' AND DATE({date_col})=DATE(?)
                """,
                (today,),
            ).fetchone()
            return int(row[0] or 0), float(row[1] or 0.0)

        try:
            q, v = _try("amount")
        except sqlite3.OperationalError as e:
            if "no such column" in str(e).lower():
                try:
                    q, v = _try("valor")
                except Exception:
                    q, v = 0, 0.0
            else:
                raise
        total_qtd += q
        total_val += v

    return total_qtd, total_val


def _sum_all_pending(cx: sqlite3.Connection) -> Tuple[int, float]:
    """
    Conta pendências EFETIVAS (dia + antigas), mesma regra do contador PowerShell:
    status='PENDENTE' OU (status='PAGA' E (paid_at IS NULL OU provider!='MP')).
    """
    total_qtd = 0
    total_val = 0.0
    for table in _existing_fee_tables(cx):
        cset = set(_cols(cx, table))
        if "status" not in cset:
            continue
        val_expr = "COALESCE(amount, valor, 0)"
        try:
            row = cx.execute(
                f"""
                SELECT COUNT(*), COALESCE(SUM({val_expr}),0)
                  FROM {table}
                 WHERE UPPER(status)='PENDENTE'
                    OR (UPPER(status)='PAGA' AND (paid_at IS NULL OR UPPER(COALESCE(provider,''))!='MP'))
                """
            ).fetchone()
            total_qtd += int(row[0] or 0)
            total_val += float(row[1] or 0.0)
        except sqlite3.OperationalError:
            row = cx.execute(
                f"""
                SELECT COUNT(*)
                  FROM {table}
                 WHERE UPPER(status)='PENDENTE'
                    OR (UPPER(status)='PAGA' AND (paid_at IS NULL OR UPPER(COALESCE(provider,''))!='MP'))
                """
            ).fetchone()
            total_qtd += int(row[0] or 0)
    return total_qtd, total_val


def _mark_paid_today(
    cx: sqlite3.Connection,
    provider: str = "MP",
    provider_status: str = "approved",
    payload_snippet: str = "",
) -> None:
    """Marca como PAGA apenas as pendências de HOJE (compatibilidade)."""
    today = _today_local()
    for table in _existing_fee_tables(cx):
        date_col = _fee_date_col(cx, table)
        if not date_col:
            continue
        set_extras = []
        cols = _cols(cx, table)
        if "paid_at" in cols:
            set_extras.append("paid_at=datetime('now','localtime')")
        if "provider" in cols:
            set_extras.append("provider=?")
        if "provider_status" in cols:
            set_extras.append("provider_status=?")
        if "provider_payload_snippet" in cols:
            set_extras.append("provider_payload_snippet=?")
        extras_sql = (", " + ", ".join(set_extras)) if set_extras else ""

        sql = f"""
            UPDATE {table}
               SET status='PAGA' {extras_sql}
             WHERE UPPER(status)='PENDENTE' AND DATE({date_col})=DATE(?)
        """
        params: List[object] = []
        if "provider" in cols:
            params.append(provider)
        if "provider_status" in cols:
            params.append(provider_status)
        if "provider_payload_snippet" in cols:
            params.append(payload_snippet[:400])
        params.append(today)
        cx.execute(sql, tuple(params))

        try:
            ids = [r[0] for r in cx.execute(f"SELECT id FROM {table} WHERE DATE({date_col})=DATE(?)", (today,)).fetchall()]
            for rid in ids:
                _reseal_row(cx, table, rid)
        except Exception:
            pass


def _mark_paid_all(
    cx: sqlite3.Connection,
    provider: str = "MP",
    provider_status: str = "approved",
    payload_snippet: str = "",
) -> None:
    """Marca como PAGA **todas** as pendências efetivas (hoje e antigas)."""
    for table in _existing_fee_tables(cx):
        cols = _cols(cx, table)
        if "status" not in cols:
            continue
        set_extras = []
        if "paid_at" in cols:
            set_extras.append("paid_at=datetime('now','localtime')")
        if "provider" in cols:
            set_extras.append("provider=?")
        if "provider_status" in cols:
            set_extras.append("provider_status=?")
        if "provider_payload_snippet" in cols:
            set_extras.append("provider_payload_snippet=?")
        extras_sql = (", " + ", ".join(set_extras)) if set_extras else ""
        sql = f"""
            UPDATE {table}
               SET status='PAGA' {extras_sql}
             WHERE UPPER(status)='PENDENTE'
                OR (UPPER(status)='PAGA' AND (paid_at IS NULL OR UPPER(COALESCE(provider,''))!='MP'))
        """
        params: List[object] = []
        if "provider" in cols:
            params.append(provider)
        if "provider_status" in cols:
            params.append(provider_status)
        if "provider_payload_snippet" in cols:
            params.append(payload_snippet[:400])
        cx.execute(sql, tuple(params))
        # Reseal defensivo
        try:
            ids = [r[0] for r in cx.execute(f"SELECT id FROM {table}").fetchall()]
            for rid in ids:
                _reseal_row(cx, table, rid)
        except Exception:
            pass


def _get_setting(cx: sqlite3.Connection, key: str, default: str = "") -> str:
    r = cx.execute("SELECT value FROM settings WHERE key=?", (key,)).fetchone()
    return r[0] if r and r[0] is not None else default


def _set_setting(cx: sqlite3.Connection, key: str, value: str) -> None:
    cx.execute(
        "INSERT INTO settings(key,value) VALUES(?,?) ON CONFLICT(key) DO UPDATE SET value=excluded.value",
        (key, value),
    )


def _parse_hhmm(s: str) -> dtime:
    try:
        h, m = map(int, (s or "18:00").split(":")[:2])
        return dtime(h, m, 0)
    except Exception:
        return dtime(18, 0, 0)


# ---------- Backfill automático (gera taxas faltantes de hoje) ----------


def _sales_sources(cx: sqlite3.Connection) -> List[Tuple[str, str]]:
    out: List[Tuple[str, str]] = []
    candidates = [("vendas", "created_at"), ("sales", "ts")]
    fallback_cols = ("created_at", "data_hora", "datetime", "data", "timestamp", "ts")
    for t, defcol in candidates:
        if _has_table(cx, t):
            cols = _cols(cx, t)
            if defcol in cols:
                out.append((t, defcol))
            else:
                for c in fallback_cols:
                    if c in cols:
                        out.append((t, c))
                        break
    return out


def _count_sales_today(cx: sqlite3.Connection, sources: List[Tuple[str, str]]) -> int:
    tot = 0
    today = _today_local()
    for t, col in sources:
        tot += cx.execute(f"SELECT COUNT(*) FROM {t} WHERE DATE({col})=DATE(?)", (today,)).fetchone()[0]
    return int(tot or 0)


def _count_fee_rows_today(cx: sqlite3.Connection) -> int:
    today = _today_local()
    tot = 0
    for table in _existing_fee_tables(cx):
        date_col = _fee_date_col(cx, table)
        if not date_col:
            continue
        tot += int(cx.execute(f"SELECT COUNT(*) FROM {table} WHERE DATE({date_col})=DATE(?)", (today,)).fetchone()[0] or 0)
    return tot


def _insert_fee_row(cx: sqlite3.Connection, amount: float) -> None:
    _ensure_min_tables(cx)
    tcols = set(_cols(cx, "taxas_cobrancas"))
    tinfo = _table_info_dict(cx, "taxas_cobrancas")  # {col: (notnull, default)}

    cols: List[str] = []
    vals: List[object] = []

    cols.append("created_at")
    vals.append(datetime.now().isoformat(timespec="seconds"))

    cols.append("status")
    vals.append("PENDENTE")

    if "amount" in tcols:
        cols.append("amount")
        vals.append(amount)
    if "valor" in tcols:
        cols.append("valor")
        vals.append(amount)

    for legacy in ("copia_cola", "txid"):
        if legacy in tcols:
            notnull, dflt = tinfo.get(legacy, (0, None))
            if notnull and (dflt is None):
                cols.append(legacy)
                vals.append("")

    if "payment_id" in tcols:
        cols.append("payment_id")
        vals.append("")

    placeholders = ",".join("?" for _ in cols)
    collist = ",".join(cols)
    cx.execute(f"INSERT INTO taxas_cobrancas ({collist}) VALUES ({placeholders})", tuple(vals))
    try:
        row_id = cx.execute("SELECT last_insert_rowid()").fetchone()[0]
        _reseal_row(cx, "taxas_cobrancas", row_id)
    except Exception:
        pass


def _auto_backfill_today(cx: sqlite3.Connection, per_sale_amount: float = 0.10) -> None:
    sources = _sales_sources(cx)
    if not sources:
        return
    sales = _count_sales_today(cx, sources)
    fees = _count_fee_rows_today(cx)
    missing = max(0, sales - fees)
    for _ in range(missing):
        _insert_fee_row(cx, per_sale_amount)


# ============================ Hora de rede (informativo) ============================


def _from_http_date(date_str: str):
    try:
        return datetime.strptime(date_str, "%a, %d %b %Y %H:%M:%S GMT").replace(tzinfo=timezone.utc)
    except Exception:
        return None


def _net_time_utc():
    """Retorna datetime em UTC usando internet, ou None se offline."""
    if not _REQ_OK:
        return None
    try:
        r = requests.get("https://worldtimeapi.org/api/timezone/America/Sao_Paulo", timeout=6)
        if r.status_code == 200:
            data = r.json()
            ut = data.get("unixtime")
            if isinstance(ut, int):
                return datetime.fromtimestamp(ut, tz=timezone.utc)
    except Exception:
        pass
    for url in ("https://www.google.com", "https://www.cloudflare.com", "https://github.com"):
        try:
            r = requests.head(url, timeout=6)
            ds = r.headers.get("Date")
            if ds:
                dt = _from_http_date(ds)
                if dt:
                    return dt
        except Exception:
            continue
    return None


def _to_sp(utc_dt: datetime) -> datetime:
    return utc_dt.astimezone(SAO_PAULO_TZ)


# ============================ Mercado Pago ============================


def _random_txid(prefix: str = "FD") -> str:
    import secrets

    alphabet = string.ascii_uppercase + string.digits
    return (prefix + "".join(secrets.choice(alphabet) for _ in range(12)))[:25]


def create_pix_charge(total: float) -> Tuple[str, str, Optional[bytes]]:
    if not _REQ_OK:
        raise RuntimeError("Biblioteca 'requests' não instalada.")
    if not MP_ACCESS_TOKEN:
        raise RuntimeError("ACCESS_TOKEN do Mercado Pago não configurado.")
    if total < MP_MIN_AMOUNT:
        raise RuntimeError(f"Valor abaixo do mínimo do provedor ({brl(MP_MIN_AMOUNT)}).")

    url = f"{MP_API_BASE}/v1/payments"
    headers = {
        "Authorization": f"Bearer {MP_ACCESS_TOKEN}",
        "Content-Type": "application/json",
        "X-Idempotency-Key": _random_txid("FD"),
    }
    body = {
        "transaction_amount": round(float(total), 2),
        "description": "Taxas da plataforma (FabricaDigital.shop)",
        "payment_method_id": "pix",
        "payer": {"email": "pagador@fabricadigital.shop"},
        "external_reference": _random_txid("FD"),
    }
    try:
        r = requests.post(url, headers=headers, json=body, timeout=MP_TIMEOUT)
        if r.status_code >= 400:
            headers["X-Idempotency-Key"] = _random_txid("FD")
            r = requests.post(url, headers=headers, json=body, timeout=MP_TIMEOUT)
        if r.status_code >= 400:
            snippet = r.text[:400].replace("\n", " ")
            raise RuntimeError(f"HTTP {r.status_code} na criação: {snippet}")
        data = r.json()
    except requests.RequestException as e:
        raise RuntimeError(f"Falha de rede/API: {e}") from e
    except ValueError:
        raise RuntimeError("Resposta inválida do provedor.")

    try:
        poi = data.get("point_of_interaction", {}).get("transaction_data", {})
        payload = poi.get("qr_code")
        qr_b64 = poi.get("qr_code_base64") or ""
        payment_id = str(data.get("id") or "")
        png_bytes = None
        if qr_b64:
            import base64

            png_bytes = base64.b64decode(qr_b64.split(",")[-1])
        if not payload or not payment_id:
            raise RuntimeError("Provedor não retornou QR/Payment ID.")
        return payload, payment_id, png_bytes
    except Exception as e:
        raise RuntimeError(f"Erro ao interpretar resposta: {e}")


def _get_payment_info(payment_id: str) -> Tuple[str, str]:
    """Retorna (status_provider, snippet_ate_400chars)."""
    if not _REQ_OK or not MP_ACCESS_TOKEN or not payment_id:
        return "", ""
    try:
        r = requests.get(
            f"{MP_API_BASE}/v1/payments/{payment_id}",
            headers={"Authorization": f"Bearer {MP_ACCESS_TOKEN}"},
            timeout=MP_TIMEOUT,
        )
        status = ""
        snippet = ""
        if r.status_code < 400:
            status = (r.json().get("status") or "").lower()
            txt = r.text or ""
            snippet = (txt[:400] if isinstance(txt, str) else "")
        return status, snippet
    except Exception:
        return "", ""


def check_paid(payment_id: str) -> bool:
    if not _REQ_OK or not MP_ACCESS_TOKEN or not payment_id:
        return False
    try:
        r = requests.get(
            f"{MP_API_BASE}/v1/payments/{payment_id}",
            headers={"Authorization": f"Bearer {MP_ACCESS_TOKEN}"},
            timeout=MP_TIMEOUT,
        )
        if r.status_code >= 400:
            return False
        return (r.json().get("status") or "").lower() == "approved"
    except Exception:
        return False


# ============================ Estilos ============================


def _init_styles() -> None:
    style = ttk.Style()
    try:
        style.theme_use(style.theme_use())
    except Exception:
        pass
    style.configure("Title.TLabel", font=("Segoe UI", 13, "bold"))
    style.configure("KPI.TLabel", font=("Segoe UI", 12, "bold"))
    style.configure("Hint.TLabel", foreground="#6b7280")
    style.configure("Status.TLabel", font=("Segoe UI", 10))
    style.configure("Danger.TLabel", foreground="#b91c1c", font=("Segoe UI", 10, "bold"))
    style.configure("Success.TLabel", foreground="#166534", font=("Segoe UI", 10, "bold"))


# ============================ UI ============================


class FeesWindow:
    def __init__(self, app, get_conn: Callable[[], sqlite3.Connection]):
        self.app = app
        self.get_conn = get_conn

        self.root = tk.Toplevel(app.root)
        self.root.title("Taxas da Plataforma")
        self.root.geometry("760x900")
        self.root.resizable(False, False)
        try:
            self.root.transient(app.root)
        except Exception:
            pass

        _init_styles()

        # estado
        self._payment_id: Optional[str] = None
        self._poll_job = None
        self._poll_deadline: Optional[datetime] = None
        self._qr_imgtk: Optional[ImageTk.PhotoImage] = None if _QR_OK else None
        self._qr_png_bytes: Optional[bytes] = None
        self._payload = ""
        self._net_status_job = None

        # layout base
        outer = ttk.Frame(self.root, padding=12)
        outer.pack(fill="both", expand=True)
        ttk.Label(outer, text="Taxas da Plataforma", style="Title.TLabel").pack(anchor="w", pady=(0, 6))

        # Agendamento (fixo — visual)
        top = ttk.LabelFrame(outer, text="Agendamento diário (fixo)", padding=10)
        top.pack(fill="x", pady=(0, 8))

        row = ttk.Frame(top)
        row.pack(fill="x")
        ttk.Label(row, text="Cobrar às (HH:MM):").pack(side="left")
        self.ent_hhmm = ttk.Entry(row, width=20, state="disabled")
        self.ent_hhmm.pack(side="left", padx=(6, 10))

        # >>>>>>>>>>>> TEXTO ATUALIZADO (fluxo novo) <<<<<<<<<<<<
        hint_txt = (
            "Horário fixo validado pela internet. Após esse horário, o PDV exige quitação "
            "quando as pendências atingirem o mínimo de R$ 10,00. "
            "Se estiver offline, há tolerância de 24h; após esse período, o bloqueio só ocorre "
            "se as pendências forem iguais ou superiores a R$ 10,00."
        )
        self.lbl_hint = ttk.Label(top, text=hint_txt, style="Hint.TLabel", wraplength=700, justify="left")
        self.lbl_hint.pack(anchor="w", pady=(6, 0))

        self.lbl_net = ttk.Label(top, text="", style="Hint.TLabel", wraplength=700, justify="left")
        self.lbl_net.pack(anchor="w", pady=(6, 0))

        # KPIs do dia
        box = ttk.LabelFrame(outer, text="Pendências de hoje", padding=10)
        box.pack(fill="x")
        krow = ttk.Frame(box)
        krow.pack(fill="x")
        self.lbl_qtd = ttk.Label(krow, text="Taxas pendentes (hoje): 0", style="KPI.TLabel")
        self.lbl_total = ttk.Label(krow, text="Total a pagar (hoje): R$ 0,00", style="KPI.TLabel")
        self.lbl_qtd.pack(side="left", padx=(0, 18))
        self.lbl_total.pack(side="left")

        # KPIs gerais (todas as datas)
        box_all = ttk.LabelFrame(outer, text="Pendências gerais (todas)", padding=10)
        box_all.pack(fill="x", pady=(8, 0))
        arow = ttk.Frame(box_all)
        arow.pack(fill="x")
        self.lbl_all_qtd = ttk.Label(arow, text="Pendências totais: 0", style="KPI.TLabel")
        self.lbl_all_total = ttk.Label(arow, text="Total estimado: R$ 0,00", style="KPI.TLabel")
        self.lbl_all_qtd.pack(side="left", padx=(0, 18))
        self.lbl_all_total.pack(side="left")

        # observação
        self.lbl_note = ttk.Label(outer, text="", style="Hint.TLabel", wraplength=700, justify="left")
        self.lbl_note.pack(anchor="w", pady=(6, 0))

        # status
        self.var_status = tk.StringVar(value="Pronto.")
        self.lbl_status = ttk.Label(outer, textvariable=self.var_status, style="Status.TLabel", wraplength=700, justify="left")
        self.lbl_status.pack(anchor="w", pady=(6, 0))

        # ações
        abtn = ttk.Frame(outer)
        abtn.pack(fill="x", pady=(6, 8))
        self.btn_generate = ttk.Button(abtn, text="Gerar cobrança PIX", command=self._generate_now)
        self.btn_generate.pack(side="left")

        # QR
        qrbox = ttk.LabelFrame(outer, text="Cobrança PIX", padding=10)
        qrbox.pack(fill="both", expand=True)
        self.lbl_qr = ttk.Label(qrbox)
        self.lbl_qr.pack(pady=(6, 10))
        self.txt = tk.Text(qrbox, height=4, wrap="word")
        self.txt.configure(state="disabled")
        self.txt.pack(fill="x")
        b2 = ttk.Frame(qrbox)
        b2.pack(pady=(8, 0))
        self.btn_copy = ttk.Button(b2, text="Copiar código", command=self._copy)
        self.btn_save = ttk.Button(b2, text="Salvar QR (PNG)", command=self._save_png)
        self.btn_copy.pack(side="left")
        self.btn_save.pack(side="left", padx=8)

        # inicialização
        self._load_hour()
        self._refresh()
        self._start_net_status_loop()

    # ---------- helpers UI ----------
    def _set_status(self, msg: str, ok: bool | None = None) -> None:
        self.var_status.set(msg)
        if ok is True:
            self.lbl_status.configure(style="Success.TLabel")
        elif ok is False:
            self.lbl_status.configure(style="Danger.TLabel")
        else:
            self.lbl_status.configure(style="Status.TLabel")

    def _disable_actions(self, on: bool) -> None:
        state = "disabled" if on else "normal"
        for b in (self.btn_generate, self.btn_copy, self.btn_save):
            try:
                b.configure(state=state)
            except Exception:
                pass

    def _set_text(self, s: str) -> None:
        self.txt.configure(state="normal")
        self.txt.delete("1.0", "end")
        self.txt.insert("1.0", s or "")
        self.txt.configure(state="disabled")
        self._payload = s or ""

    def _copy(self) -> None:
        if not self._payload:
            return
        try:
            self.root.clipboard_clear()
            self.root.clipboard_append(self._payload)
            messagebox.showinfo("Taxas — PIX", "Código PIX copiado para a área de transferência.")
        except Exception:
            pass

    def _save_png(self) -> None:
        if not _QR_OK:
            return
        if self._qr_png_bytes:
            path = filedialog.asksaveasfilename(
                defaultextension=".png", initialfile="taxas_pix.png", filetypes=[("PNG", "*.png")]
            )
            if not path:
                return
            try:
                with open(path, "wb") as f:
                    f.write(self._qr_png_bytes)
                messagebox.showinfo("Taxas — PIX", f"QR salvo em:\n{path}")
            except Exception as e:
                messagebox.showerror("Taxas — PIX", f"Falha ao salvar PNG:\n{e}")
        elif self._payload:
            path = filedialog.asksaveasfilename(
                defaultextension=".png", initialfile="taxas_pix.png", filetypes=[("PNG", "*.png")]
            )
            if not path:
                return
            try:
                img = qrcode.make(self._payload).resize((420, 420))
                img.save(path)
                messagebox.showinfo("Taxas — PIX", f"QR salvo em:\n{path}")
            except Exception as e:
                messagebox.showerror("Taxas — PIX", f"Falha ao salvar PNG:\n{e}")

    def _render_qr(self) -> None:
        if not _QR_OK:
            self.lbl_qr.configure(image="")
            self._qr_imgtk = None
            return
        try:
            if self._qr_png_bytes:
                from io import BytesIO

                img = Image.open(BytesIO(self._qr_png_bytes)).resize((420, 420))
            elif self._payload:
                img = qrcode.make(self._payload).resize((420, 420))
            else:
                self.lbl_qr.configure(image="")
                self._qr_imgtk = None
                return
            self._qr_imgtk = ImageTk.PhotoImage(img)  # type: ignore[call-arg]
            self.lbl_qr.configure(image=self._qr_imgtk)
        except Exception:
            self.lbl_qr.configure(image="")
            self._qr_imgtk = None

    # ---------- dados ----------
    def _load_hour(self) -> None:
        self.ent_hhmm.configure(state="normal")
        self.ent_hhmm.delete(0, "end")
        self.ent_hhmm.insert(0, CUTOFF_FIXED_STR)
        self.ent_hhmm.configure(state="disabled")
        try:
            with self.get_conn() as cx:
                _ensure_min_tables(cx)
                _set_setting(cx, "taxes_hour", "00:00")
        except Exception:
            pass

    def _save_hour(self) -> None:
        try:
            with self.get_conn() as cx:
                _ensure_min_tables(cx)
                _set_setting(cx, "taxes_hour", "00:00")
        except Exception:
            pass

    def _refresh(self) -> None:
        try:
            self._save_hour()
            with self.get_conn() as cx:
                _ensure_min_tables(cx)
                _auto_backfill_today(cx, per_sale_amount=0.10)
                n_today, total_today = _sum_pending_today(cx)
                n_all, total_all = _sum_all_pending(cx)
        except Exception as e:
            n_today, total_today, n_all, total_all = 0, 0.0, 0, 0.0
            print("[fees] refresh error:", e)

        self.lbl_qtd.config(text=f"Taxas pendentes (hoje): {n_today}")
        self.lbl_total.config(text=f"Total a pagar (hoje): {brl(total_today)}")
        self.lbl_all_qtd.config(text=f"Pendências totais: {n_all}")
        self.lbl_all_total.config(text=f"Total estimado: {brl(total_all)}")

        if total_today < MP_MIN_AMOUNT:
            self.lbl_note.config(
                text=f"Valor do dia abaixo do mínimo ({brl(MP_MIN_AMOUNT)}). O PDV segue liberado; as taxas acumulam até atingir o mínimo."
            )
        else:
            self.lbl_note.config(text="")

        if n_today == 0 or total_today <= 0:
            self._qr_png_bytes = None
            self._payload = ""
            self._render_qr()
            self._set_text("Sem pendências de hoje.")
            self._stop_poll()
            self._payment_id = None
            self._set_status("Pronto.")
            self._disable_actions(False)

        # status da hora de rede inicia/continua por loop
        # (chamado no start loop)

    # ---------- status da hora de rede / offline grace ----------
    def _start_net_status_loop(self) -> None:
        self._update_net_status()
        self._net_status_job = self.root.after(15_000, self._start_net_status_loop)

    def _update_net_status(self) -> None:
        utc = _net_time_utc()
        if utc:
            local = _to_sp(utc)
            try:
                with self.get_conn() as cx:
                    _ensure_min_tables(cx)
                    _set_setting(cx, "NET_TIME_LAST_OK_ISO", utc.isoformat())
                    _set_setting(cx, "OFFLINE_GRACE_START_ISO", "")
            except Exception:
                pass
            msg = f"Hora de rede OK — São Paulo: {local.strftime('%Y-%m-%d %H:%M:%S')}  |  Corte: 00:00 (fixo)"
            self.lbl_net.config(text=msg)
        else:
            try:
                with self.get_conn() as cx:
                    _ensure_min_tables(cx)
                    started = _get_setting(cx, "OFFLINE_GRACE_START_ISO", "")
                    if not started:
                        now_local = datetime.now(SAO_PAULO_TZ)
                        _set_setting(cx, "OFFLINE_GRACE_START_ISO", now_local.isoformat())
                        left_h = OFFLINE_GRACE_HOURS
                    else:
                        try:
                            st = datetime.fromisoformat(started)
                            if st.tzinfo is None:
                                st = st.replace(tzinfo=SAO_PAULO_TZ)
                        except Exception:
                            st = datetime.now(SAO_PAULO_TZ)
                        elapsed = datetime.now(SAO_PAULO_TZ) - st
                        left = max(0, OFFLINE_GRACE_HOURS * 3600 - int(elapsed.total_seconds()))
                        h, rem = divmod(left, 3600)
                        m, _ = divmod(rem, 60)
                        left_h = f"{h:02d}:{m:02d}h"
                msg = (
                    f"Sem internet — uso liberado por até 24h (restante: {left_h}). "
                    "Conecte-se para validar a hora e quitar as taxas."
                )
                self.lbl_net.config(text=msg)
            except Exception:
                self.lbl_net.config(text="Sem internet — uso liberado por até 24h.")

    # ---------- ações ----------
    def _generate_now(self) -> None:
        try:
            with self.get_conn() as cx:
                _ensure_min_tables(cx)
                n_all, total_all = _sum_all_pending(cx)  # cobra TODAS as pendências
        except Exception as e:
            messagebox.showerror("Taxas — PIX", f"Falha ao calcular pendências:\n{e}")
            return

        if n_all == 0 or total_all <= 0:
            messagebox.showinfo("Taxas — PIX", "Não há pendências para cobrar.")
            return
        if total_all < MP_MIN_AMOUNT:
            messagebox.showinfo(
                "Taxas — PIX",
                f"Total geral abaixo do mínimo ({brl(MP_MIN_AMOUNT)}). O PDV está liberado; as taxas vão acumular até atingir o mínimo.",
            )
            self._set_status("Aguardando acumular até o mínimo…")
            return

        self._set_status("Gerando cobrança PIX (todas as pendências)…")
        self._disable_actions(True)

        payload = None
        payment_id = None
        png_bytes = None
        try:
            payload, payment_id, png_bytes = create_pix_charge(total_all)
        except Exception as e:
            self._set_status("Falha ao criar cobrança dinâmica. Verifique sua conexão/credenciais.", ok=False)
            messagebox.showerror("PIX (Mercado Pago)", str(e))
            self._disable_actions(False)
            self._payment_id = None
            self._poll_deadline = None
            return

        self._payload = payload or ""
        self._qr_png_bytes = png_bytes
        self._set_text(self._payload)
        self._render_qr()

        # compat: associar payment_id às linhas de hoje em taxas_cobrancas (se existir coluna)
        if payment_id:
            try:
                with self.get_conn() as cx:
                    cx.execute(
                        """
                        UPDATE taxas_cobrancas
                           SET payment_id=?
                         WHERE UPPER(status)='PENDENTE'
                           AND DATE(created_at)=DATE(?)
                           AND (payment_id IS NULL OR payment_id='')
                        """,
                        (payment_id, _today_local()),
                    )
                    try:
                        ids = [
                            r[0]
                            for r in cx.execute(
                                """
                                SELECT id FROM taxas_cobrancas
                                 WHERE UPPER(status)='PENDENTE'
                                   AND DATE(created_at)=DATE(?)
                                """,
                                (_today_local(),),
                            ).fetchall()
                        ]
                        for rid in ids:
                            _reseal_row(cx, "taxas_cobrancas", rid)
                    except Exception:
                        pass
            except Exception as e:
                print("[fees] warn: persist payment_id:", e)

        self._payment_id = payment_id
        if payment_id:
            self._poll_deadline = datetime.now() + timedelta(minutes=15)
            self._set_status("Cobrança criada. Aguardando pagamento…")
            self._start_poll()
        else:
            self._poll_deadline = None
            self._set_status("Cobrança pronta (sem polling).")
            self._disable_actions(False)

    # ---------- polling ----------
    def _start_poll(self) -> None:
        self._stop_poll()
        self._tick_poll()

    def _tick_poll(self) -> None:
        if self._poll_deadline:
            left = int((self._poll_deadline - datetime.now()).total_seconds())
            if left <= 0:
                self._set_status("Tempo esgotado. Gere outra cobrança.", ok=False)
                self._disable_actions(False)
                self._payment_id = None
                self._poll_deadline = None
                return
            m, s = divmod(left, 60)
            self._set_status(f"Aguardando pagamento… ({m:02d}:{s:02d})")
        self._poll_job = self.root.after(POLL_INTERVAL_S * 1000, self._poll_once)

    def _stop_poll(self) -> None:
        if self._poll_job:
            try:
                self.root.after_cancel(self._poll_job)
            except Exception:
                pass
        self._poll_job = None

    def _poll_once(self) -> None:
        pid = self._payment_id
        if pid and check_paid(pid):
            self._on_paid()
            return
        self._tick_poll()

    def _on_paid(self) -> None:
        provider_status, snippet = _get_payment_info(self._payment_id or "")
        try:
            with self.get_conn() as cx:
                _mark_paid_all(
                    cx,
                    provider="MP",
                    provider_status=(provider_status or "approved"),
                    payload_snippet=(snippet or ""),
                )
        except Exception as e:
            print("[fees] mark paid error:", e)
        self._stop_poll()
        self._payment_id = None
        self._poll_deadline = None
        self._set_status("Pagamento confirmado! Todas as pendências marcadas como PAGA.", ok=True)
        self._disable_actions(False)
        self._refresh()


# ============================ API pública ============================


def open_fees(app, get_conn: Callable[[], sqlite3.Connection]):
    return FeesWindow(app, get_conn)
