# fees_block.py
from __future__ import annotations
import sqlite3
from datetime import datetime, timedelta, timezone, time as dtime
from tkinter import messagebox
from decimal import Decimal, ROUND_HALF_UP

# ---------- Política ----------
MP_MIN_AMOUNT = Decimal("10.00")  # mínimo real do provedor — usado no bloqueio por valor (antes: 1.00)
OFFLINE_GRACE_HOURS = 24          # tolerância sem internet (h)
SAO_PAULO_TZ = timezone(timedelta(hours=-3))
CUTOFF_LABEL = "00:00 (fixo, hora de rede)"

# ---------- Dependência opcional de rede ----------
try:
    import requests
    _REQ_OK = True
except Exception:
    _REQ_OK = False

# ---------- Selo HMAC ----------
import hmac, hashlib
# >>>>>> TROQUE PARA A MESMA CHAVE USADA NO migration_security.py E fees.py <<<<<<
_HMAC_SECRET = b'\xcc\x8cW\xa1R\xe3r|3qFt\x8e`\xd6v\t\x0f\x12\x82\xef\x1c\xb1\xf0\x0b\xb6\xd5a\x01\xa6\xa0;'
_SEAL_FIELDS = ["id","created_at","amount","status","sale_id","payment_id","paid_at","provider","provider_status"]

def _seal_from_row_dict(d: dict) -> str:
    def _n(v):
        if v is None: return ""
        s = str(v).strip()
        return s.replace("\r","").replace("\n"," ")
    payload = "|".join(_n(d.get(k)) for k in _SEAL_FIELDS).encode("utf-8")
    return hmac.new(_HMAC_SECRET, payload, hashlib.sha256).hexdigest()

# ---------- Conexão / Settings ----------
def _conn(db_path: str):
    cx = sqlite3.connect(db_path, timeout=30, isolation_level=None)
    cx.row_factory = sqlite3.Row
    return cx

def _ensure_settings(cx: sqlite3.Connection):
    cx.execute("""
        CREATE TABLE IF NOT EXISTS settings(
            key   TEXT PRIMARY KEY,
            value TEXT
        )
    """)

def _get_setting(cx, key: str):
    _ensure_settings(cx)
    r = cx.execute("SELECT value FROM settings WHERE key=?", (key,)).fetchone()
    return (r["value"] if isinstance(r, sqlite3.Row) else (r[0] if r else None)) if r else None

def _set_setting(cx, key: str, value: str):
    _ensure_settings(cx)
    cx.execute(
        "INSERT INTO settings(key,value) VALUES(?,?) "
        "ON CONFLICT(key) DO UPDATE SET value=excluded.value",
        (key, value)
    )

# ---------- Util legado ----------
def _parse_time(hhmm: str):
    from datetime import datetime as _dt
    try:
        h, m = map(int, (hhmm or "18:00").split(":")[:2])
        return dtime(h, m, 0)
    except Exception:
        try:
            return _dt.strptime(hhmm, "%H:%M").time()
        except Exception:
            return dtime(18, 0, 0)

# ---------- Tabelas ----------
def _table_exists(cx, name: str) -> bool:
    q = "SELECT 1 FROM sqlite_master WHERE (type='table' OR type='view') AND lower(name)=lower(?)"
    return cx.execute(q, (name,)).fetchone() is not None

def _has_column(cx, table: str, col: str) -> bool:
    try:
        cols = [r[1].lower() for r in cx.execute(f'PRAGMA table_info("{table}")')]
        return col.lower() in cols
    except Exception:
        return False

def _candidate_amount_column(cx: sqlite3.Connection, table: str) -> str | None:
    for c in ["amount", "valor", "value", "total", "fee_amount"]:
        if _has_column(cx, table, c):
            return c
    return None

# ---------- Hora de rede ----------
def _from_http_date(date_str: str):
    try:
        return datetime.strptime(date_str, "%a, %d %b %Y %H:%M:%S GMT").replace(tzinfo=timezone.utc)
    except Exception:
        return None

def _net_time_utc():
    if not _REQ_OK:
        return None
    try:
        r = requests.get("https://worldtimeapi.org/api/timezone/America/Sao_Paulo", timeout=6)
        if r.status_code == 200:
            data = r.json()
            ut = data.get("unixtime")
            if isinstance(ut, int):
                return datetime.fromtimestamp(ut, tz=timezone.utc)
    except Exception:
        pass
    for url in ("https://www.google.com", "https://www.cloudflare.com", "https://github.com"):
        try:
            r = requests.head(url, timeout=6)
            ds = r.headers.get("Date")
            if ds:
                dt = _from_http_date(ds)
                if dt: return dt
        except Exception:
            continue
    return None

def _to_sp(utc_dt: datetime) -> datetime:
    return utc_dt.astimezone(SAO_PAULO_TZ)

# ---------- Integridade (tamper) ----------
def _any_tamper(cx: sqlite3.Connection) -> bool:
    """
    Verifica selo HMAC SOMENTE quando a tabela possui coluna 'seal'.
    Linhas sem 'seal' (registros antigos/migração) NÃO disparam tamper.
    """
    try:
        for table in ("taxas_cobrancas","platform_fees"):
            if not _table_exists(cx, table):
                continue
            if not _has_column(cx, table, "seal"):
                # Sem coluna de seal -> não avalia tamper nesta tabela
                continue

            cur = cx.execute(f"SELECT * FROM {table}")
            for row in cur.fetchall():
                try:
                    d = dict(row)
                except Exception:
                    d = {}
                s_db = (str(d.get("seal") or "")).strip()
                if not s_db:
                    # registro sem seal (antigo) -> ignora
                    continue

                # completa campos esperados
                for k in _SEAL_FIELDS:
                    if k not in d or d[k] is None:
                        d[k] = ""

                s_calc = _seal_from_row_dict(d)
                if s_db != s_calc:
                    return True  # selo inválido
        return False
    except Exception:
        return True  # falhou ao checar => bloqueia por segurança

# ---------- Helpers ----------
def _to_decimal(v) -> Decimal:
    if v is None:
        return Decimal("0")
    if isinstance(v, (int, float)):
        return Decimal(str(v))
    s = str(v).strip().replace(",", ".")
    try:
        return Decimal(s)
    except Exception:
        return Decimal("0")

# Soma das pendências reais (status='PENDENTE' apenas)
def _sum_pending_amount_strict(cx: sqlite3.Connection) -> Decimal:
    total = Decimal("0")
    for table in ("taxas_cobrancas", "platform_fees"):
        if not _table_exists(cx, table):
            continue
        amount_col = _candidate_amount_column(cx, table)
        if not amount_col:
            continue
        try:
            cur = cx.execute(
                f"SELECT {amount_col} FROM {table} WHERE UPPER(status)='PENDENTE'"
            )
            for (val,) in cur.fetchall():
                total += _to_decimal(val)
        except Exception:
            pass
    return total.quantize(Decimal("0.01"), rounding=ROUND_HALF_UP)

# ---------- Regras principais ----------
def _offline_grace_ok(cx: sqlite3.Connection) -> bool:
    """
    Controle da janela de tolerância offline (24h).
    """
    started_iso = _get_setting(cx, "OFFLINE_GRACE_START_ISO") or ""
    if not started_iso:
        now_local = datetime.now(SAO_PAULO_TZ)
        _set_setting(cx, "OFFLINE_GRACE_START_ISO", now_local.isoformat())
        return True
    try:
        started = datetime.fromisoformat(started_iso)
        if started.tzinfo is None:
            started = started.replace(tzinfo=SAO_PAULO_TZ)
    except Exception:
        started = datetime.now(SAO_PAULO_TZ) - timedelta(hours=OFFLINE_GRACE_HOURS + 1)
    now_local = datetime.now(SAO_PAULO_TZ)
    return (now_local - started) <= timedelta(hours=OFFLINE_GRACE_HOURS)

def must_block_for_fees(db_path: str) -> bool:
    """
    Bloqueia se:
      - ONLINE: (tamper) OU (pendente >= MP_MIN_AMOUNT)
      - OFFLINE:
          * <= 24h: libera
          * > 24h: bloqueia apenas se pendente >= MP_MIN_AMOUNT
    """
    with _conn(db_path) as cx:
        _ensure_settings(cx)
        utc_now = _net_time_utc()

        if utc_now is not None:
            # Online: zera janela offline e avalia tamper + valor
            _set_setting(cx, "NET_TIME_LAST_OK_ISO", utc_now.isoformat())
            _set_setting(cx, "OFFLINE_GRACE_START_ISO", "")

            if _any_tamper(cx):
                return True

            try:
                pend_amount = _sum_pending_amount_strict(cx)
            except Exception:
                pend_amount = Decimal("0.00")
            return pend_amount >= MP_MIN_AMOUNT

        # Sem internet
        if _offline_grace_ok(cx):
            return False  # dentro da tolerância: libera sempre

        # Offline > 24h: só bloqueia se atingiu o mínimo
        try:
            pend_amount = _sum_pending_amount_strict(cx)
        except Exception:
            pend_amount = Decimal("0.00")
        return pend_amount >= MP_MIN_AMOUNT

def guard_block(db_path: str, parent_window=None) -> bool:
    """
    Retorna True para BLOQUEAR ou False para LIBERAR.
    Mostra mensagens adequadas para cada caso.
    """
    # texto do mínimo (formato BR)
    min_txt = f"R$ {MP_MIN_AMOUNT:.2f}".replace(".", ",")

    with _conn(db_path) as cx:
        blocked = must_block_for_fees(db_path)
        if blocked:
            utc_now = _net_time_utc()
            if utc_now is None:
                # Offline e janela expirou: aqui só bloqueia se pendências >= mínimo
                try:
                    messagebox.showerror(
                        "Modo offline expirado",
                        f"Você está sem internet há mais de 24 horas e há taxas pendentes "
                        f"em valor igual ou superior a {min_txt}.\n"
                        "Conecte-se à internet e quite as taxas para liberar novas vendas.",
                        parent=parent_window
                    )
                except Exception:
                    pass
                return True

            # Online: tamper OU pendências >= mínimo
            try:
                messagebox.showwarning(
                    "Pagamento de Taxas",
                    f"Há taxas pendentes em valor igual ou superior a {min_txt}, "
                    "ou registros com integridade comprometida.\n"
                    "Efetue o pagamento das taxas para liberar novas vendas.",
                    parent=parent_window
                )
            except Exception:
                pass
            return True

        # liberado mas offline (janela ativa): aviso de tolerância
        if _net_time_utc() is None:
            try:
                messagebox.showwarning(
                    "Modo offline",
                    "Sem internet — uso liberado por até 24h de tolerância.\n"
                    "Ao reconectar, eventuais taxas acumuladas precisarão estar quitadas.",
                    parent=parent_window
                )
            except Exception:
                pass
        return False
