# fees_utils.py
import sqlite3
import uuid
from datetime import datetime

# --- util interno ---
def _colinfo(conn: sqlite3.Connection, table: str):
    cur = conn.execute(f'PRAGMA table_info("{table}")')
    # (cid, name, type, notnull, dflt_value, pk)
    cols = [(r[1].lower(), r[3] == 1) for r in cur.fetchall()]
    names = {n for (n, _) in cols}
    notnull = {n: nn for (n, nn) in cols}
    return names, notnull

def _has_col(conn, table, col):
    names, _ = _colinfo(conn, table)
    return col.lower() in names

def _col_notnull(conn, table, col):
    _, notnull = _colinfo(conn, table)
    return bool(notnull.get(col.lower(), False))

def _ensure_table(conn: sqlite3.Connection):
    """
    Garante a tabela base. NÃO remove colunas antigas (ex.: valor, copia_cola, txid…).
    Apenas cria/ajusta a estrutura mínima caso não exista.
    """
    conn.execute("""
        CREATE TABLE IF NOT EXISTS taxas_cobrancas(
          id INTEGER PRIMARY KEY AUTOINCREMENT,
          created_at TEXT NOT NULL,
          status TEXT NOT NULL DEFAULT 'PENDENTE',
          amount REAL,
          sale_id INTEGER,
          payment_id TEXT
        )
    """)

    # === Acrescenta colunas que porventura faltem (bancos antigos) ===
    names, _ = _colinfo(conn, "taxas_cobrancas")

    if "status" not in names:
        conn.execute("ALTER TABLE taxas_cobrancas ADD COLUMN status TEXT NOT NULL DEFAULT 'PENDENTE'")

    # amount: se nem 'amount' nem 'valor' existem, adiciona 'amount'
    if "amount" not in names and "valor" not in names:
        conn.execute("ALTER TABLE taxas_cobrancas ADD COLUMN amount REAL")

    if "sale_id" not in names:
        conn.execute("ALTER TABLE taxas_cobrancas ADD COLUMN sale_id INTEGER")

    if "payment_id" not in names:
        conn.execute("ALTER TABLE taxas_cobrancas ADD COLUMN payment_id TEXT")

    # Migra e sincroniza amount <-> valor se a coluna antiga existir
    names, _ = _colinfo(conn, "taxas_cobrancas")
    if "valor" in names and "amount" not in names:
        conn.execute("ALTER TABLE taxas_cobrancas ADD COLUMN amount REAL")
        conn.execute("UPDATE taxas_cobrancas SET amount=COALESCE(valor,0.10)")
    elif "valor" in names and "amount" in names:
        conn.execute("UPDATE taxas_cobrancas SET amount=COALESCE(amount, valor)")
        conn.execute("UPDATE taxas_cobrancas SET valor =COALESCE(valor , amount)")

    # Saneia NOT NULL antigos (copia_cola / txid) se houverem e tiverem NULL
    for col in ("copia_cola", "txid"):
        if _has_col(conn, "taxas_cobrancas", col) and _col_notnull(conn, "taxas_cobrancas", col):
            conn.execute(f"UPDATE taxas_cobrancas SET {col}='' WHERE {col} IS NULL")

    # Índices úteis
    conn.execute("CREATE INDEX IF NOT EXISTS idx_taxas_created ON taxas_cobrancas(substr(created_at,1,10))")
    conn.execute("CREATE INDEX IF NOT EXISTS idx_taxas_sale ON taxas_cobrancas(sale_id)")
    conn.execute("CREATE INDEX IF NOT EXISTS idx_taxas_status ON taxas_cobrancas(status)")

    # Garante 1 taxa por venda (sem quebrar bancos antigos):
    # índice único PARCIAL (permite múltiplos NULLs, mas impede duplicar sale_id válido)
    try:
        conn.execute("""
            CREATE UNIQUE INDEX IF NOT EXISTS idx_taxa_por_venda
                ON taxas_cobrancas(sale_id)
             WHERE sale_id IS NOT NULL
        """)
    except Exception:
        # versões muito antigas do SQLite podem não suportar índice parcial; ignorar silenciosamente
        pass

def _unique_txid(prefix: str = "FEE") -> str:
    # microsegundos + uuid reduz colisão mesmo com UNIQUE(txid)
    return f"{prefix}-{datetime.now().strftime('%Y%m%d%H%M%S%f')}-{uuid.uuid4().hex[:8]}"

# --- wrapper de retry para coluna ausente (banco muito antigo) ---
def _retry_add_sale_id(fn):
    def wrap(conn: sqlite3.Connection, *args, **kwargs):
        try:
            return fn(conn, *args, **kwargs)
        except sqlite3.OperationalError as e:
            msg = str(e).lower()
            if "no such column: sale_id" in msg:
                # adiciona a coluna e tenta novamente
                conn.execute("ALTER TABLE taxas_cobrancas ADD COLUMN sale_id INTEGER")
                conn.execute("CREATE INDEX IF NOT EXISTS idx_taxas_sale ON taxas_cobrancas(sale_id)")
                try:
                    conn.execute("""
                        CREATE UNIQUE INDEX IF NOT EXISTS idx_taxa_por_venda
                            ON taxas_cobrancas(sale_id)
                         WHERE sale_id IS NOT NULL
                    """)
                except Exception:
                    pass
                return fn(conn, *args, **kwargs)
            raise
    return wrap

# --- API pública ---
def ensure_tax_table(conn: sqlite3.Connection) -> None:
    _ensure_table(conn)

@_retry_add_sale_id
def insert_fee(conn: sqlite3.Connection, *, sale_id: int, status: str, amount: float = 0.10,
               txid: str | None = None, copia_cola: str = "") -> int:
    """
    Insere uma taxa por venda. Tolera esquemas com 'valor', 'copia_cola', 'txid' (NOT NULL/UNIQUE) etc.
    Retorna fee_id.
    """
    _ensure_table(conn)
    names, notnull = _colinfo(conn, "taxas_cobrancas")

    cols = ["created_at", "status", "sale_id"]
    vals = [datetime.now().isoformat(), status, int(sale_id)]

    # amount / valor
    if "amount" in names:
        cols.append("amount"); vals.append(float(amount))
    if "valor" in names:
        cols.append("valor"); vals.append(float(amount))

    # txid (preencher se existe e/ou se é NOT NULL/UNIQUE)
    if "txid" in names:
        txid_final = txid or _unique_txid("FEE")
        cols.append("txid"); vals.append(txid_final)

    # copia_cola (se existir e for NOT NULL, preencher ao menos com vazio)
    if "copia_cola" in names:
        if notnull.get("copia_cola", False) and (copia_cola is None):
            copia_cola = ""
        cols.append("copia_cola"); vals.append(copia_cola or "")

    sql = f"INSERT INTO taxas_cobrancas({','.join(cols)}) VALUES({','.join('?'*len(vals))})"
    cur = conn.execute(sql, vals)
    return int(cur.lastrowid)

@_retry_add_sale_id
def mark_fee_paid_by_sale(conn: sqlite3.Connection, sale_id: int, *, payment_id: str | None = None) -> int:
    """
    Marca todas as taxas dessa venda como PAGA. Retorna número de linhas afetadas.
    """
    _ensure_table(conn)
    if payment_id:
        cur = conn.execute(
            "UPDATE taxas_cobrancas SET status='PAGA', payment_id=? WHERE sale_id=? AND status!='PAGA'",
            (payment_id, int(sale_id))
        )
    else:
        cur = conn.execute(
            "UPDATE taxas_cobrancas SET status='PAGA' WHERE sale_id=? AND status!='PAGA'",
            (int(sale_id),)
        )
    return int(cur.rowcount)

@_retry_add_sale_id
def mark_fee_paid_by_txid(conn: sqlite3.Connection, txid: str, *, payment_id: str | None = None) -> int:
    """
    Caso você use txid para casar cobrança, use este. Retorna linhas afetadas.
    """
    _ensure_table(conn)
    if payment_id:
        cur = conn.execute(
            "UPDATE taxas_cobrancas SET status='PAGA', payment_id=? WHERE txid=? AND status!='PAGA'",
            (payment_id, txid)
        )
    else:
        cur = conn.execute(
            "UPDATE taxas_cobrancas SET status='PAGA' WHERE txid=? AND status!='PAGA'",
            (txid,)
        )
    return int(cur.rowcount)

# --- Relatórios/auxiliares (opcionais) ---
def fees_today_summary(conn: sqlite3.Connection) -> dict:
    """
    Totais do DIA ATUAL (horário local).
    Retorna: count_all, sum_all, count_pending, sum_pending
    """
    _ensure_table(conn)
    row_all = conn.execute("""
        SELECT COUNT(*) AS cnt, COALESCE(SUM(amount),0) AS total
          FROM taxas_cobrancas
         WHERE date(created_at) = date('now','localtime')
    """).fetchone()
    row_pend = conn.execute("""
        SELECT COUNT(*) AS cnt, COALESCE(SUM(amount),0) AS total
          FROM taxas_cobrancas
         WHERE status='PENDENTE'
           AND date(created_at) = date('now','localtime')
    """).fetchone()
    return {
        "count_all": int(row_all[0] or 0),
        "sum_all": float(row_all[1] or 0.0),
        "count_pending": int(row_pend[0] or 0),
        "sum_pending": float(row_pend[1] or 0.0),
    }

def ensure_fees_for_all_sales(conn: sqlite3.Connection, *, default_amount: float = 0.10) -> int:
    """
    BACKFILL opcional: cria uma taxa para cada venda da tabela 'sales' que ainda não tenha taxa.
    Útil se houveram vendas feitas antes da instalação deste módulo.
    Retorna número de taxas criadas.
    """
    _ensure_table(conn)
    # só segue se a tabela 'sales' existir
    has_sales = conn.execute("""
        SELECT 1 FROM sqlite_master WHERE type='table' AND lower(name)='sales'
    """).fetchone()
    if not has_sales:
        return 0

    created = 0
    # Seleciona vendas sem taxa associada
    sql = """
        SELECT s.id, s.payment_method
          FROM sales s
     LEFT JOIN taxas_cobrancas t
            ON t.sale_id = s.id
         WHERE t.id IS NULL
    """
    rows = conn.execute(sql).fetchall()
    for r in rows:
        sid = int(r[0])
        pm  = (r[1] or "").lower()
        # PAGA para cash/card; PENDENTE para pix
        status = "PAGA" if pm in ("cash", "card") else "PENDENTE"
        try:
            insert_fee(conn, sale_id=sid, status=status, amount=float(default_amount))
            created += 1
        except sqlite3.IntegrityError:
            # se houver UNIQUE por sale_id e já existir, ignora silenciosamente
            pass
    return created
