# -*- coding: utf-8 -*-
# fiscal_config_dialog.py
import sqlite3
import tkinter as tk
from tkinter import ttk, messagebox, filedialog
from typing import Optional

UF_LIST = ["AC","AL","AP","AM","BA","CE","DF","ES","GO","MA","MT",
           "MS","MG","PA","PB","PR","PE","PI","RJ","RN","RS",
           "RO","RR","SC","SP","SE","TO"]

def _ensure_table_and_row(get_conn):
    with get_conn() as con:
        con.execute("""
        CREATE TABLE IF NOT EXISTS fiscal_config (
          id INTEGER PRIMARY KEY CHECK(id=1),
          mode TEXT NOT NULL CHECK(mode IN ('RECIBO','NFCE')),
          cnpj TEXT, ie TEXT, regime TEXT, uf TEXT,
          ambiente TEXT CHECK(ambiente IN ('homolog','prod')),
          csc_id TEXT, csc_token TEXT,
          cert_pfx BLOB, cert_password TEXT,
          serie INTEGER DEFAULT 1, numero INTEGER DEFAULT 1,
          updated_at TEXT
        )
        """)
        row = con.execute("SELECT 1 FROM fiscal_config WHERE id=1").fetchone()
        if not row:
            con.execute("""
            INSERT INTO fiscal_config
            (id, mode, cnpj, ie, regime, uf, ambiente, csc_id, csc_token, cert_pfx, cert_password, serie, numero, updated_at)
            VALUES (1,'NFCE','','','','SP','homolog','','',NULL,'',1,1,datetime('now'))
            """)

def _digits_only(s: str) -> str:
    return "".join(ch for ch in (s or "") if ch.isdigit())

def _load_current(get_conn) -> dict:
    _ensure_table_and_row(get_conn)
    with get_conn() as con:
        con.row_factory = sqlite3.Row
        r = con.execute("SELECT * FROM fiscal_config WHERE id=1").fetchone()
    return dict(r) if r else {}

def _save_config(get_conn, data: dict):
    _ensure_table_and_row(get_conn)
    with get_conn() as con:
        con.execute("""
            UPDATE fiscal_config SET
              mode='NFCE',
              cnpj=?, ie=?, regime=?, uf=?,
              ambiente=?, csc_id=?, csc_token=?,
              cert_password=?, serie=?, numero=?, updated_at=datetime('now')
            WHERE id=1
        """, (
            data.get("cnpj",""), data.get("ie",""), data.get("regime",""),
            data.get("uf","SP"), data.get("ambiente","homolog"),
            data.get("csc_id",""), data.get("csc_token",""),
            data.get("cert_password",""), int(data.get("serie") or 1),
            int(data.get("numero") or 1),
        ))

def _save_cert_blob(get_conn, blob: bytes):
    _ensure_table_and_row(get_conn)
    with get_conn() as con:
        con.execute("UPDATE fiscal_config SET cert_pfx=?, updated_at=datetime('now') WHERE id=1", (blob,))

def open_fiscal_config_dialog(app, get_conn):
    # UI
    win = tk.Toplevel(app.root)
    win.title("Configurar Nota (NFC-e)")
    win.transient(app.root)
    win.grab_set()
    win.resizable(False, False)

    pad = (12, 8)
    frm = ttk.Frame(win, padding=12)
    frm.pack(fill="both", expand=True)

    cur = _load_current(get_conn)

    var_cnpj = tk.StringVar(value=cur.get("cnpj",""))
    var_ie   = tk.StringVar(value=cur.get("ie",""))
    var_reg  = tk.StringVar(value=cur.get("regime","simples") or "simples")
    var_uf   = tk.StringVar(value=cur.get("uf","SP") or "SP")
    var_amb  = tk.StringVar(value=cur.get("ambiente","homolog") or "homolog")
    var_cscid   = tk.StringVar(value=cur.get("csc_id",""))
    var_csctok  = tk.StringVar(value=cur.get("csc_token",""))
    var_pwd  = tk.StringVar(value=cur.get("cert_password",""))
    var_serie = tk.StringVar(value=str(cur.get("serie",1)))
    var_num   = tk.StringVar(value=str(cur.get("numero",1)))

    row = 0
    ttk.Label(frm, text="CNPJ (somente números):").grid(row=row, column=0, sticky="w", padx=pad, pady=(10,3))
    ttk.Entry(frm, textvariable=var_cnpj, width=28).grid(row=row, column=1, sticky="w", padx=pad, pady=(10,3)); row+=1

    ttk.Label(frm, text="IE (somente números, vazio se isento):").grid(row=row, column=0, sticky="w", padx=pad, pady=3)
    ttk.Entry(frm, textvariable=var_ie, width=28).grid(row=row, column=1, sticky="w", padx=pad, pady=3); row+=1

    ttk.Label(frm, text="Regime:").grid(row=row, column=0, sticky="w", padx=pad, pady=3)
    cb_reg = ttk.Combobox(frm, values=["simples","normal"], textvariable=var_reg, width=25, state="readonly")
    cb_reg.grid(row=row, column=1, sticky="w", padx=pad, pady=3); row+=1

    ttk.Label(frm, text="UF:").grid(row=row, column=0, sticky="w", padx=pad, pady=3)
    cb_uf = ttk.Combobox(frm, values=UF_LIST, textvariable=var_uf, width=25, state="readonly")
    cb_uf.grid(row=row, column=1, sticky="w", padx=pad, pady=3); row+=1

    ttk.Label(frm, text="Ambiente:").grid(row=row, column=0, sticky="w", padx=pad, pady=3)
    amb = ttk.Frame(frm); amb.grid(row=row, column=1, sticky="w", padx=pad, pady=3)
    ttk.Radiobutton(amb, text="Homologação", value="homolog", variable=var_amb).pack(side="left")
    ttk.Radiobutton(amb, text="Produção",    value="prod",    variable=var_amb).pack(side="left"); row+=1

    ttk.Separator(frm).grid(row=row, column=0, columnspan=2, sticky="ew", padx=12, pady=(8,8)); row+=1

    ttk.Label(frm, text="CSC ID:").grid(row=row, column=0, sticky="w", padx=pad, pady=3)
    ttk.Entry(frm, textvariable=var_cscid, width=28).grid(row=row, column=1, sticky="w", padx=pad, pady=3); row+=1

    ttk.Label(frm, text="CSC Token:").grid(row=row, column=0, sticky="w", padx=pad, pady=3)
    ttk.Entry(frm, textvariable=var_csctok, width=28).grid(row=row, column=1, sticky="w", padx=pad, pady=3); row+=1

    ttk.Separator(frm).grid(row=row, column=0, columnspan=2, sticky="ew", padx=12, pady=(8,8)); row+=1

    ttk.Label(frm, text="Série:").grid(row=row, column=0, sticky="w", padx=pad, pady=3)
    ttk.Entry(frm, textvariable=var_serie, width=10).grid(row=row, column=1, sticky="w", padx=pad, pady=3); row+=1

    ttk.Label(frm, text="Próximo número:").grid(row=row, column=0, sticky="w", padx=pad, pady=3)
    ttk.Entry(frm, textvariable=var_num, width=10).grid(row=row, column=1, sticky="w", padx=pad, pady=3); row+=1

    ttk.Separator(frm).grid(row=row, column=0, columnspan=2, sticky="ew", padx=12, pady=(8,8)); row+=1

    ttk.Label(frm, text="Senha do certificado A1 (.pfx):").grid(row=row, column=0, sticky="w", padx=pad, pady=3)
    ent_pwd = ttk.Entry(frm, textvariable=var_pwd, width=28, show="•")
    ent_pwd.grid(row=row, column=1, sticky="w", padx=pad, pady=3); row+=1

    def do_load_pfx():
        path = filedialog.askopenfilename(title="Selecionar certificado A1 (.pfx)",
                                          filetypes=[("Certificado A1", "*.pfx"), ("Todos", "*.*")],
                                          parent=win)
        if not path:
            return
        try:
            blob = open(path, "rb").read()
            _save_cert_blob(get_conn, blob)
            messagebox.showinfo("Config NFC-e", "Certificado A1 importado com sucesso.", parent=win)
        except Exception as e:
            messagebox.showerror("Config NFC-e", f"Falha ao importar certificado:\n{e}", parent=win)

    ttk.Button(frm, text="Importar certificado (.pfx)…", command=do_load_pfx).grid(row=row, column=1, sticky="w", padx=pad, pady=(0,8)); row+=1

    # Botões
    btns = ttk.Frame(frm); btns.grid(row=row, column=0, columnspan=2, sticky="e", padx=12, pady=(10,0))

    def do_save():
        # validações leves
        cnpj = _digits_only(var_cnpj.get())
        if len(cnpj) not in (14,):
            messagebox.showerror("Config NFC-e", "CNPJ deve ter 14 dígitos.", parent=win); return
        ie = _digits_only(var_ie.get())
        try:
            serie = int(var_serie.get() or "1")
            numero = int(var_num.get() or "1")
        except Exception:
            messagebox.showerror("Config NFC-e", "Série e número devem ser inteiros.", parent=win); return
        regime = (var_reg.get() or "simples").lower()
        if regime not in ("simples","normal"):
            messagebox.showerror("Config NFC-e", "Regime deve ser 'simples' ou 'normal'.", parent=win); return
        uf = var_uf.get().upper()
        if uf not in UF_LIST:
            messagebox.showerror("Config NFC-e", "UF inválida.", parent=win); return
        amb = var_amb.get()
        if amb not in ("homolog","prod"):
            messagebox.showerror("Config NFC-e", "Ambiente inválido.", parent=win); return

        data = dict(
            cnpj=cnpj, ie=ie, regime=regime, uf=uf, ambiente=amb,
            csc_id=(var_cscid.get() or "").strip(),
            csc_token=(var_csctok.get() or "").strip(),
            cert_password=(var_pwd.get() or "").strip(),
            serie=serie, numero=numero
        )
        try:
            _save_config(get_conn, data)
            messagebox.showinfo("Config NFC-e", "Configuração salva com sucesso.", parent=win)
        except Exception as e:
            messagebox.showerror("Config NFC-e", f"Falha ao salvar:\n{e}", parent=win)

    ttk.Button(btns, text="Salvar", command=do_save).pack(side="right", padx=(6,0))
    ttk.Button(btns, text="Fechar", command=win.destroy).pack(side="right")
