#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
manual_builder.py — Gera docs/manual.md a partir do código Tkinter.
Uso:
    python tools/manual_builder.py --root . --out docs/manual.md
"""
from __future__ import annotations
import argparse, ast, json
from dataclasses import dataclass, field
from pathlib import Path
from typing import Dict, List, Optional

@dataclass
class MenuItemDoc:
    label: str
    command: Optional[str] = None
    file: str = ""

@dataclass
class MenuDoc:
    name: str
    items: List[MenuItemDoc] = field(default_factory=list)

@dataclass
class WindowDoc:
    class_name: str
    base_class: str
    path: str
    doc: Optional[str]

@dataclass
class ProjectDoc:
    menus: Dict[str, MenuDoc] = field(default_factory=dict)
    windows: List[WindowDoc] = field(default_factory=list)
    callbacks_docs: Dict[str, str] = field(default_factory=dict)

def _get_str(node: ast.AST) -> Optional[str]:
    if isinstance(node, ast.Constant) and isinstance(node.value, str):
        return node.value.strip()
    if isinstance(node, ast.Str):
        return node.s.strip()
    return None

def _extract_kwargs(call: ast.Call) -> Dict[str, str]:
    out: Dict[str, str] = {}
    for kw in call.keywords:
        k = kw.arg
        if k is None:
            continue
        v = _get_str(kw.value)
        if v is None and isinstance(kw.value, ast.Name):
            v = kw.value.id
        elif v is None and isinstance(kw.value, ast.Attribute):
            try:
                base = kw.value.value.id if isinstance(kw.value.value, ast.Name) else ""
                v = f"{base}.{kw.value.attr}".strip(".")
            except Exception:
                v = kw.value.attr
        out[k] = v
    return out

def _is_tk_window_base(b: ast.expr) -> bool:
    targets = {
        ("tk", "Toplevel"), ("tkinter", "Toplevel"),
        ("tkinter", "Frame"), ("ttk", "Frame"), ("tk", "Frame"),
    }
    if isinstance(b, ast.Attribute) and isinstance(b.value, ast.Name):
        return (b.value.id, b.attr) in targets
    if isinstance(b, ast.Name):
        return b.id in {"Toplevel", "Frame"}
    return False

def _base_name(b: ast.expr) -> str:
    if isinstance(b, ast.Attribute) and isinstance(b.value, ast.Name):
        return f"{b.value.id}.{b.attr}"
    if isinstance(b, ast.Name):
        return b.id
    try:
        return ast.unparse(b)
    except Exception:
        return str(b)

def parse_project(root: Path) -> ProjectDoc:
    proj = ProjectDoc()
    ignore_dirs = {"__pycache__", ".venv", "venv", "env", ".git", "site-packages", "build", "dist"}
    py_files = []
    for p in root.rglob("*.py"):
        rel = p.relative_to(root)
        if any(part in ignore_dirs for part in rel.parts):
            continue
        py_files.append(p)
    for fpath in py_files:
        try:
            src = fpath.read_text(encoding="utf-8", errors="ignore")
            tree = ast.parse(src, filename=str(fpath))
        except Exception:
            continue
        menu_var_to_name: Dict[str, Optional[str]] = {}
        class Visitor(ast.NodeVisitor):
            def visit_FunctionDef(self, node: ast.FunctionDef):
                doc = ast.get_docstring(node)
                if doc:
                    proj.callbacks_docs[node.name] = doc.strip()
                self.generic_visit(node)
            def visit_ClassDef(self, node: ast.ClassDef):
                base_hit = None
                for b in node.bases:
                    if _is_tk_window_base(b):
                        base_hit = _base_name(b); break
                if base_hit:
                    doc = ast.get_docstring(node)
                    proj.windows.append(WindowDoc(
                        class_name=node.name,
                        base_class=base_hit,
                        path=str(fpath.relative_to(root)),
                        doc=doc.strip() if doc else None,
                    ))
                self.generic_visit(node)
            def visit_Assign(self, node: ast.Assign):
                try:
                    if isinstance(node.value, ast.Call) and isinstance(node.value.func, ast.Attribute):
                        if node.value.func.attr == "Menu":
                            for t in node.targets:
                                if isinstance(t, ast.Name):
                                    menu_var_to_name.setdefault(t.id, None)
                except Exception:
                    pass
                self.generic_visit(node)
            def visit_Call(self, node: ast.Call):
                if isinstance(node.func, ast.Attribute):
                    method = node.func.attr
                    if method in {"add_cascade","add_command"}:
                        kwargs = _extract_kwargs(node)
                        label = kwargs.get("label")
                        cmd = kwargs.get("command")
                        if method == "add_cascade":
                            mvar = kwargs.get("menu")
                            if mvar and mvar in menu_var_to_name:
                                menu_var_to_name[mvar] = label or menu_var_to_name.get(mvar) or mvar
                            if label:
                                proj.menus.setdefault(label, MenuDoc(name=label))
                        else:
                            parent_name = "Desconhecido"
                            try:
                                if isinstance(node.func.value, ast.Name):
                                    cand = node.func.value.id
                                    bound = menu_var_to_name.get(cand)
                                    if bound: parent_name = bound
                            except Exception:
                                pass
                            if parent_name not in proj.menus:
                                proj.menus[parent_name] = MenuDoc(name=parent_name)
                            proj.menus[parent_name].items.append(MenuItemDoc(
                                label=label or "(sem rótulo)",
                                command=cmd,
                                file=str(fpath.relative_to(root))
                            ))
                self.generic_visit(node)
        Visitor().visit(tree)
    return proj

def build_markdown(proj: ProjectDoc) -> str:
    lines = []
    lines.append("# Manual do Usuário — ERP/PDV\n")
    lines.append("_Gerado automaticamente a partir do código (menus, janelas, docstrings)._")
    lines.append("")
    if proj.menus:
        lines.append("## Menus e Funcionalidades")
        for mname, mdoc in sorted(proj.menus.items(), key=lambda kv: kv[0].lower()):
            lines.append(f"### {mname}")
            if not mdoc.items:
                lines.append("- (Sem itens detectados)")
            else:
                for item in mdoc.items:
                    if item.command and item.command in proj.callbacks_docs:
                        first_line = proj.callbacks_docs[item.command].splitlines()[0]
                        desc = first_line
                    elif item.command:
                        desc = f"Ação: `{item.command}`"
                    else:
                        desc = "Ação não identificada."
                    lines.append(f"- **{item.label}** — {desc}  \n  _Fonte: {item.file}_")
            lines.append("")
    else:
        lines.append("## Menus e Funcionalidades\n(Não foram detectados menus Tkinter via `add_cascade`/`add_command`.)\n")
    lines.append("## Janelas (Toplevel/Frame)")
    if proj.windows:
        for w in sorted(proj.windows, key=lambda w: w.class_name.lower()):
            doc = w.doc or "_Sem descrição — adicione docstring na classe para enriquecer o manual._"
            lines.append(f"### {w.class_name} ({w.base_class})\n**Arquivo:** `{w.path}`\n\n{doc}\n")
    else:
        lines.append("(Nenhuma classe de janela detectada.)\n")
    return "\n".join(lines)

def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--root", default=".", help="Pasta raiz do projeto")
    ap.add_argument("--out", default="docs/manual.md", help="Arquivo de saída (markdown)")
    ap.add_argument("--index", default="docs/manual_index.json", help="Arquivo JSON com índice auxiliar")
    args = ap.parse_args()
    root = Path(args.root).resolve()
    out_md = Path(args.out).resolve()
    out_md.parent.mkdir(parents=True, exist_ok=True)
    proj = parse_project(root)
    md = build_markdown(proj)
    out_md.write_text(md, encoding="utf-8")
    index = {
        "menus": {
            k: [{"label": it.label, "command": it.command, "file": it.file} for it in v.items]
            for k, v in proj.menus.items()
        },
        "windows": [w.__dict__ for w in proj.windows],
        "callbacks_with_doc": list(proj.callbacks_docs.keys())
    }
    Path(args.index).resolve().write_text(json.dumps(index, ensure_ascii=False, indent=2), encoding="utf-8")
    print(f"Manual gerado em: {out_md}")
    print(f"Índice auxiliar em: {args.index}")

if __name__ == "__main__":
    main()
