#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Abre o manual do usuário no navegador:
1) Prioriza docs/manual.html (estilizado).
2) Se não existir, renderiza docs/manual.md em HTML leve.
Corrigido: não usamos str.format() no template para não conflitar com chaves do CSS.
"""
from __future__ import annotations
from pathlib import Path
import tempfile
import webbrowser
import html
import sys
import os
import tkinter as tk
from tkinter import messagebox
from typing import Optional

HTML_REL_PATH = Path("docs") / "manual.html"
MD_REL_PATH   = Path("docs") / "manual.md"

# Usamos um marcador {{BODY}} e substituímos via .replace() para não quebrar o CSS.
HTML_WRAPPER = """<!doctype html>
<html lang="pt-BR">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title>Manual do Usuário — ERP/PDV</title>
<style>
:root{--bg:#fff;--fg:#1f2937;--muted:#6b7280;--border:#e5e7eb;--surface:#f9fafb;--brand:#0ea5e9;--brand2:#0369a1}
@media (prefers-color-scheme: dark){
:root{--bg:#0b1021;--fg:#e5e7eb;--muted:#9ca3af;--border:#1f2937;--surface:#0f172a;--brand:#38bdf8;--brand2:#0ea5e9}
}
*{box-sizing:border-box}
body{margin:0;background:var(--bg);color:var(--fg);
font:15px/1.6 -apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,"Helvetica Neue",Arial,"Noto Sans","Liberation Sans",sans-serif}
.container{max-width:1000px;margin:0 auto;padding:24px}
.card{background:#fff1;border:1px solid var(--border);border-radius:16px;padding:20px}
h1{font-size:26px;margin:0 0 10px} h2{font-size:20px;margin:14px 0 6px} h3{font-size:16px;margin:12px 0 6px}
a{color:var(--brand);text-decoration:none} a:hover{text-decoration:underline}
.code{background:#0002;border:1px solid var(--border);padding:12px 14px;border-radius:12px;overflow:auto;font-family:ui-monospace,Menlo,Consolas,monospace;font-size:13px}
hr{border:0;height:1px;background:var(--border);margin:16px 0}
ul,ol{margin:6px 0 6px 20px}
pre{white-space:pre-wrap;word-wrap:break-word}
</style>
</head>
<body>
  <div class="container">
    {{BODY}}
  </div>
</body>
</html>
"""

def _try_import_markdown():
    try:
        import markdown  # type: ignore
        return markdown
    except Exception:
        return None

def _md_to_html(md_text: str) -> str:
    """Converte Markdown para HTML. Se 'markdown' não estiver presente, usa <pre>."""
    md = _try_import_markdown()
    if md:
        content = md.markdown(md_text, extensions=["extra", "sane_lists", "toc"])
        body = f"<div class='card'>{content}</div>"
    else:
        safe = html.escape(md_text)
        body = f"<h1>Manual do Usuário — ERP/PDV</h1><div class='card'><pre>{safe}</pre></div>"
    return HTML_WRAPPER.replace("{{BODY}}", body)

def _open_url(url: str) -> None:
    webbrowser.open(url)

def _open_with_system(path: Path) -> bool:
    try:
        if sys.platform.startswith("win"):
            os.startfile(str(path))  # type: ignore[attr-defined]
            return True
        elif sys.platform == "darwin":
            os.system(f"open '{path}'")
            return True
        else:
            os.system(f"xdg-open '{path}'")
            return True
    except Exception:
        return False

def open_manual(base_dir: Path, root: Optional[tk.Tk] = None) -> None:
    """
    Abre o manual.
    - Se existir docs/manual.html: abre direto no navegador.
    - Senão, tenta docs/manual.md -> gera HTML temporário e abre.
    - Último fallback: abre o .md pelo SO; se falhar, mostra messagebox.
    """
    html_path = (base_dir / HTML_REL_PATH).resolve()
    if html_path.exists():
        _open_url(html_path.as_uri())
        return

    md_path = (base_dir / MD_REL_PATH).resolve()
    if md_path.exists():
        md_text = md_path.read_text(encoding="utf-8", errors="ignore")
        html_content = _md_to_html(md_text)
        with tempfile.NamedTemporaryFile("w", suffix=".html", delete=False, encoding="utf-8") as f:
            f.write(html_content)
            tmp = Path(f.name)
        _open_url(tmp.as_uri())
        return

    if _open_with_system(md_path):
        return

    if root is not None:
        messagebox.showerror("Manual", f"Manual não encontrado em:\n{html_path}\n{md_path}")
    else:
        raise FileNotFoundError(f"Manual não encontrado em: {html_path} ou {md_path}")
