# migrar_tudo.py
import sys, sqlite3, datetime

DB = sys.argv[1] if len(sys.argv) > 1 else "erp.sqlite3"
SEED = "--seed" in sys.argv

con = sqlite3.connect(DB)
cur = con.cursor()
con.execute("PRAGMA foreign_keys = ON")

def table_exists(name: str) -> bool:
    return cur.execute("SELECT 1 FROM sqlite_master WHERE type='table' AND name=?", (name,)).fetchone() is not None

def columns(name: str):
    if not table_exists(name): return set()
    return {r[1].lower() for r in cur.execute(f"PRAGMA table_info({name})")}

def ensure_table(sql: str):
    cur.execute(sql)

def ensure_column(tbl: str, col: str, ddl: str):
    if not table_exists(tbl): return
    if col.lower() not in columns(tbl):
        try:
            cur.execute(ddl)
            print(f"[OK] {tbl}.{col} criado")
        except Exception as e:
            msg = str(e).lower()
            if "duplicate column name" in msg or "already exists" in msg:
                pass
            else:
                raise

print(f"== Migrando banco: {DB}")

# ---------------- Base do PDV ----------------
# products
ensure_table("""
CREATE TABLE IF NOT EXISTS products (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  sku TEXT,
  name TEXT NOT NULL,
  unit_price REAL NOT NULL DEFAULT 0
)""")
# colunas opcionais
ensure_column("products", "barcode", "ALTER TABLE products ADD COLUMN barcode TEXT")
ensure_column("products", "stock_qty", "ALTER TABLE products ADD COLUMN stock_qty REAL DEFAULT 0")

# sales
ensure_table("""
CREATE TABLE IF NOT EXISTS sales (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  customer_id INTEGER,
  ts TEXT NOT NULL,
  total_gross REAL NOT NULL,
  discount_pct REAL NOT NULL DEFAULT 0,
  payment_method TEXT NOT NULL,
  received REAL NOT NULL DEFAULT 0,
  "change" REAL NOT NULL DEFAULT 0
)""")
ensure_column("sales", "status", "ALTER TABLE sales ADD COLUMN status TEXT DEFAULT 'PENDENTE'")

# sale_items
ensure_table("""
CREATE TABLE IF NOT EXISTS sale_items (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  sale_id INTEGER NOT NULL,
  product_id INTEGER NOT NULL,
  qty INTEGER NOT NULL,
  unit_price REAL NOT NULL,
  FOREIGN KEY(sale_id) REFERENCES sales(id) ON DELETE CASCADE
)""")

# cash_registers (para Caixa)
ensure_table("""
CREATE TABLE IF NOT EXISTS cash_registers (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  opened_at TEXT NOT NULL,
  opening_amount REAL NOT NULL DEFAULT 0,
  operator_open TEXT,
  notes TEXT,
  closed_at TEXT,
  closing_amount REAL DEFAULT 0,
  operator_close TEXT
)""")

# cash_movements (sangria/suprimento) — opcional
ensure_table("""
CREATE TABLE IF NOT EXISTS cash_movements (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  created_at TEXT NOT NULL,
  type TEXT NOT NULL CHECK(type IN ('IN','OUT')),
  method TEXT NOT NULL,       -- CASH, PIX, etc.
  amount REAL NOT NULL DEFAULT 0,
  reason TEXT,                -- SUPPLY/WITHDRAW
  reference_type TEXT,
  reference_id TEXT,
  notes TEXT
)""")

# settings (chave/valor)
ensure_table("""
CREATE TABLE IF NOT EXISTS settings (
  key TEXT PRIMARY KEY,
  value TEXT
)""")

# ---------------- Fiscal (NFC-e) ----------------
ensure_table("""
CREATE TABLE IF NOT EXISTS fiscal_config (
  id INTEGER PRIMARY KEY CHECK(id=1),
  mode TEXT NOT NULL CHECK(mode IN ('RECIBO','NFCE')),
  cnpj TEXT, ie TEXT, regime TEXT, uf TEXT,
  ambiente TEXT CHECK(ambiente IN ('homolog','prod')),
  csc_id TEXT, csc_token TEXT,
  cert_pfx BLOB, cert_password TEXT,
  serie INTEGER DEFAULT 1, numero INTEGER DEFAULT 1,
  updated_at TEXT
)""")

ensure_table("""
CREATE TABLE IF NOT EXISTS invoices (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  sale_id INTEGER NOT NULL,
  model TEXT NOT NULL CHECK(model IN ('NFCE')),
  chave TEXT, numero INTEGER, serie INTEGER,
  status TEXT NOT NULL CHECK(status IN ('pending','authorized','rejected','error')),
  protocolo TEXT, xml TEXT, pdf_path TEXT,
  created_at TEXT, authorized_at TEXT
)""")

ensure_table("""
CREATE TABLE IF NOT EXISTS invoice_items (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  invoice_id INTEGER NOT NULL,
  product_id INTEGER, description TEXT,
  ncm TEXT, cfop TEXT,
  qty REAL, unit TEXT, unit_price REAL, total REAL,
  tax_json TEXT
)""")

# Linha padrão da config fiscal (id=1)
row = cur.execute("SELECT 1 FROM fiscal_config WHERE id=1").fetchone()
if not row:
    cur.execute("""
    INSERT INTO fiscal_config
    (id, mode, cnpj, ie, regime, uf, ambiente, csc_id, csc_token, cert_pfx, cert_password, serie, numero, updated_at)
    VALUES (1,'NFCE','','','','SP','homolog','','',NULL,'',1,1,datetime('now'))
    """)
    print("[OK] fiscal_config seed (id=1)")

# ---------------- Seed opcional (produto + venda de teste) ----------------
if SEED:
    # produto
    r = cur.execute("SELECT id FROM products LIMIT 1").fetchone()
    if not r:
        cur.execute("INSERT INTO products (sku,name,unit_price,barcode,stock_qty) VALUES (?,?,?,?,?)",
                    ("SKU-001", "Produto de Teste", 10.00, None, 100))
        print("[OK] Produto de teste criado")
    # venda
    r = cur.execute("SELECT id FROM sales LIMIT 1").fetchone()
    if not r:
        now = datetime.datetime.now().isoformat()
        cur.execute("""INSERT INTO sales (customer_id, ts, total_gross, discount_pct, payment_method, received, "change", status)
                       VALUES (NULL, ?, 10.00, 0, 'cash', 10.00, 0, 'PAGA')""", (now,))
        sale_id = cur.lastrowid
        # pega qualquer produto
        pid = cur.execute("SELECT id FROM products ORDER BY id LIMIT 1").fetchone()[0]
        cur.execute("INSERT INTO sale_items (sale_id, product_id, qty, unit_price) VALUES (?,?,?,?)",
                    (sale_id, pid, 1, 10.00))
        print(f"[OK] Venda de teste criada: sale_id = {sale_id}")

con.commit(); con.close()
print("== Migração concluída com sucesso.")
