# migration_security.py
import sys, os, sqlite3, hmac, hashlib

# >>>>>> TROQUE ESTA CHAVE e guarde-a com você (local). <<<<<<
# Use a MESMA chave nos arquivos: migration_security.py, fees.py e fees_block.py
_HMAC_SECRET = b'\xcc\x8cW\xa1R\xe3r|3qFt\x8e`\xd6v\t\x0f\x12\x82\xef\x1c\xb1\xf0\x0b\xb6\xd5a\x01\xa6\xa0;'
# Alias para compatibilidade com chamadas existentes
HMAC_SECRET = _HMAC_SECRET

# Campos usados no selo (ordem estável)
SEAL_FIELDS = [
    "id","created_at","amount","status","sale_id","payment_id",
    "paid_at","provider","provider_status"
]

STATUS_WHITELIST = ("PENDENTE","PAGA","PENDENTE_DELETADA")
DATE_CANDIDATES = ("created_at","created","datetime","ts","data","data_hora","timestamp")

DDL_SQL = """
-- Auditoria
CREATE TABLE IF NOT EXISTS audit_taxas (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  op TEXT NOT NULL,
  table_name TEXT NOT NULL,
  old_status TEXT, new_status TEXT,
  old_amount REAL, new_amount REAL,
  ts TEXT NOT NULL DEFAULT (datetime('now','localtime'))
);

-- Tabela de whitelist de status
CREATE TABLE IF NOT EXISTS _status_whitelist(status TEXT PRIMARY KEY);
"""

TRIGGERS_SQL = {
"taxas_cobrancas": """
-- Bloqueia DELETE direto
CREATE TRIGGER IF NOT EXISTS tg_txc_no_delete
BEFORE DELETE ON taxas_cobrancas
FOR EACH ROW
BEGIN
  INSERT INTO audit_taxas(op, table_name, old_status, old_amount)
  VALUES ('DELETE','taxas_cobrancas', OLD.status, OLD.amount);
  SELECT RAISE(ABORT, 'Delete bloqueado: use regra de negócio (status).');
END;

-- Audita UPDATE de status/amount
CREATE TRIGGER IF NOT EXISTS tg_txc_audit_update
AFTER UPDATE ON taxas_cobrancas
FOR EACH ROW
WHEN (OLD.status IS NOT NEW.status) OR (COALESCE(OLD.amount,0) != COALESCE(NEW.amount,0))
BEGIN
  INSERT INTO audit_taxas(op, table_name, old_status, new_status, old_amount, new_amount)
  VALUES ('UPDATE','taxas_cobrancas', OLD.status, NEW.status, OLD.amount, NEW.amount);
END;

-- Valida status permitido
CREATE TRIGGER IF NOT EXISTS tg_txc_validate_status
BEFORE UPDATE ON taxas_cobrancas
FOR EACH ROW
WHEN (SELECT COUNT(*) FROM _status_whitelist WHERE status=NEW.status)=0
BEGIN
  SELECT RAISE(ABORT, 'Status inválido');
END;
""",
"platform_fees": """
CREATE TRIGGER IF NOT EXISTS tg_pf_no_delete
BEFORE DELETE ON platform_fees
FOR EACH ROW
BEGIN
  INSERT INTO audit_taxas(op, table_name, old_status, old_amount)
  VALUES ('DELETE','platform_fees', OLD.status, OLD.amount);
  SELECT RAISE(ABORT, 'Delete bloqueado: use regra de negócio (status).');
END;

CREATE TRIGGER IF NOT EXISTS tg_pf_audit_update
AFTER UPDATE ON platform_fees
FOR EACH ROW
WHEN (OLD.status IS NOT NEW.status) OR (COALESCE(OLD.amount,0) != COALESCE(NEW.amount,0))
BEGIN
  INSERT INTO audit_taxas(op, table_name, old_status, new_status, old_amount, new_amount)
  VALUES ('UPDATE','platform_fees', OLD.status, NEW.status, OLD.amount, NEW.amount);
END;

CREATE TRIGGER IF NOT EXISTS tg_pf_validate_status
BEFORE UPDATE ON platform_fees
FOR EACH ROW
WHEN (SELECT COUNT(*) FROM _status_whitelist WHERE status=NEW.status)=0
BEGIN
  SELECT RAISE(ABORT, 'Status inválido');
END;
"""
}

def has_table(cx, name):
    return cx.execute(
        "SELECT 1 FROM sqlite_master WHERE (type='table' OR type='view') AND lower(name)=lower(?)",
        (name,)
    ).fetchone() is not None

def cols(cx, table):
    try:
        return [r[1].lower() for r in cx.execute(f'PRAGMA table_info("{table}")')]
    except:
        return []

def ensure_col(cx, table, col, decl="TEXT"):
    if col.lower() not in cols(cx, table):
        cx.execute(f'ALTER TABLE {table} ADD COLUMN {col} {decl}')

def norm(v):
    if v is None: return ""
    s = str(v).strip()
    return s.replace("\r","").replace("\n"," ").strip()

def compute_seal(rowdict):
    parts = []
    for k in SEAL_FIELDS:
        parts.append(norm(rowdict.get(k)))
    data = "|".join(parts).encode("utf-8")
    return hmac.new(HMAC_SECRET, data, hashlib.sha256).hexdigest()

def pick_date_expr(cx, table):
    """Retorna uma expressão SQL que produza 'created_at' de forma segura."""
    c = set(cols(cx, table))
    for cand in DATE_CANDIDATES:
        if cand.lower() in c:
            return f"COALESCE({cand}, '') AS created_at"
    return "'' AS created_at"  # nenhum campo de data conhecido

def backfill_table(cx, table):
    if not has_table(cx, table): return

    # Garante colunas necessárias (compatíveis com seu app)
    for col, decl in [
        ("amount","REAL"), ("status","TEXT"), ("sale_id","INTEGER"),
        ("payment_id","TEXT"), ("paid_at","TEXT"),
        ("provider","TEXT"), ("provider_status","TEXT"),
        ("provider_payload_snippet","TEXT"),
        ("seal","TEXT")
    ]:
        try: ensure_col(cx, table, col, decl)
        except Exception: pass

    # Monta SELECT seguro (created_at pode não existir)
    created_expr = pick_date_expr(cx, table)

    sel = f"""
    SELECT
      id,
      {created_expr},
      amount,
      status,
      sale_id,
      payment_id,
      paid_at,
      provider,
      provider_status
    FROM {table}
    """
    rows = cx.execute(sel).fetchall()

    for r in rows:
        # suporte tanto para Row quanto para tuple
        if isinstance(r, sqlite3.Row):
            d = dict(r)
        else:
            d = {
                "id": r[0], "created_at": r[1], "amount": r[2], "status": r[3],
                "sale_id": r[4], "payment_id": r[5], "paid_at": r[6],
                "provider": r[7], "provider_status": r[8],
            }
        seal = compute_seal(d)
        cx.execute(f"UPDATE {table} SET seal=? WHERE id=? AND (seal IS NULL OR seal='')", (seal, d["id"]))

def main(dbpath):
    dbpath = os.path.abspath(dbpath)
    folder = os.path.dirname(dbpath) or "."
    if not os.path.isdir(folder):
        raise SystemExit(f"Pasta não existe: {folder}")
    print(f"Usando banco: {dbpath}")
    try:
        cx = sqlite3.connect(dbpath)
    except sqlite3.OperationalError as e:
        raise SystemExit(f"Falha ao abrir '{dbpath}': {e}")

    cx.row_factory = sqlite3.Row
    with cx:
        # base
        cx.executescript(DDL_SQL)
        for s in STATUS_WHITELIST:
            cx.execute("INSERT OR IGNORE INTO _status_whitelist(status) VALUES (?)", (s,))

        # por tabela
        for t in ("taxas_cobrancas","platform_fees"):
            if has_table(cx, t):
                backfill_table(cx, t)
                cx.executescript(TRIGGERS_SQL[t])
            else:
                print(f"(Aviso) Tabela não encontrada: {t}")

    print("Migração concluída com sucesso.")
    cx.close()

if __name__ == "__main__":
    if len(sys.argv) < 2:
        print("Uso: python migration_security.py caminho/do/arquivo.db")
        sys.exit(1)
    main(sys.argv[1])
