# -*- coding: utf-8 -*-
# nfce_adapter.py
import io, json, hashlib, datetime, sqlite3, pathlib, platform, os, subprocess
from dataclasses import dataclass
from typing import Dict, Any, Tuple, List
from contextlib import contextmanager

from lxml import etree
from signxml import XMLSigner, methods
from cryptography.hazmat.primitives.serialization import pkcs12
from cryptography.hazmat.backends import default_backend
import qrcode
from reportlab.lib.pagesizes import A4
from reportlab.pdfgen import canvas
from reportlab.lib.utils import ImageReader

@contextmanager
def _tx(conn: sqlite3.Connection):
    try:
        conn.execute("BEGIN")
        yield
        conn.execute("COMMIT")
    except Exception:
        try: conn.execute("ROLLBACK")
        except Exception: pass
        raise

UF_CODES = {"AC":12,"AL":27,"AP":16,"AM":13,"BA":29,"CE":23,"DF":53,"ES":32,"GO":52,"MA":21,
            "MT":51,"MS":50,"MG":31,"PA":15,"PB":25,"PR":41,"PE":26,"PI":22,"RJ":33,"RN":24,
            "RS":43,"RO":11,"RR":14,"SC":42,"SP":35,"SE":28,"TO":17}

def only_digits(s: str) -> str:
    return "".join([c for c in (s or "") if c.isdigit()])

@dataclass
class NFCeConfig:
    cnpj: str; ie: str; regime: str; uf: str; ambiente: str
    csc_id: str; csc_token: str
    serie: int; numero: int
    cert_pfx: bytes; cert_password: str

def load_config(get_conn) -> NFCeConfig:
    with get_conn() as con:
        con.row_factory = sqlite3.Row
        r = con.execute("SELECT * FROM fiscal_config WHERE id=1").fetchone()
        if not r: raise RuntimeError("Config fiscal ausente (fiscal_config.id=1).")
        return NFCeConfig(
            cnpj=r["cnpj"], ie=r["ie"], regime=r["regime"], uf=r["uf"], ambiente=r["ambiente"],
            csc_id=r["csc_id"], csc_token=r["csc_token"], serie=int(r["serie"] or 1),
            numero=int(r["numero"] or 1), cert_pfx=(r["cert_pfx"] or b""), cert_password=r["cert_password"] or ""
        )

def _pfx_to_keycert(pfx_bytes: bytes, password: str) -> Tuple[bytes, bytes]:
    key, cert, _ = pkcs12.load_key_and_certificates(
        pfx_bytes, password.encode("utf-8") if password else None, default_backend()
    )
    if not key or not cert: raise RuntimeError("Certificado A1 inválido.")
    from cryptography.hazmat.primitives.serialization import Encoding, PrivateFormat, NoEncryption
    key_pem  = key.private_bytes(Encoding.PEM, PrivateFormat.PKCS8, NoEncryption())
    cert_pem = cert.public_bytes(Encoding.PEM)
    return key_pem, cert_pem

def chave_acesso(uf: str, cnpj: str, data: datetime.datetime, serie: int, numero: int, tipo_emis: int, cNF: str) -> str:
    uf_code = UF_CODES.get((uf or "").upper(), 35)
    aamm = f"{data.year%100:02d}{data.month:02d}"
    base = f"{uf_code:02d}{aamm}{only_digits(cnpj).zfill(14)}65{serie:03d}{numero:09d}{tipo_emis:08d}{cNF.zfill(8)}"
    pesos = [2,3,4,5,6,7,8,9]*5
    soma = 0
    for i,ch in enumerate(reversed(base)): soma += int(ch)*pesos[i]
    dv = 11 - (soma % 11); dv = 0 if dv >= 10 else dv
    return base + str(dv)

def qrcode_text(chave: str, csc_id: str, csc_token: str, total: float, uf: str, ambiente: str) -> str:
    # ATENÇÃO: ajuste a URL da sua UF para produção.
    base = "https://www.sefaz.rs.gov.br/NFCE/NFCE-COM.aspx" if (uf or "").upper()=="RS" else "https://www.sefaz.fazenda.gov.br/nfce/consultar"
    params = f"p={chave}|2|{csc_id}|{total:.2f}"
    hashval = hashlib.sha1((params + csc_token).encode("utf-8")).hexdigest()
    return base + "?" + params + "|" + hashval

def _money_fmt(v: float) -> str:
    s = f"{float(v):,.2f}".replace(",", "X").replace(".", ",").replace("X",".")
    return s

def build_xml_nfce(sale: Dict[str,Any], items: List[Dict[str,Any]], cfg: NFCeConfig, chave: str) -> etree._Element:
    NFE = "{http://www.portalfiscal.inf.br/nfe}"; nsmap = {None:"http://www.portalfiscal.inf.br/nfe"}
    nfe = etree.Element(NFE+"NFe", nsmap=nsmap); inf = etree.SubElement(nfe, NFE+"infNFe", Id="NFe"+chave, versao="4.00")
    ide = etree.SubElement(inf, NFE+"ide")
    etree.SubElement(ide, NFE+"cUF").text = str(UF_CODES.get(cfg.uf.upper(),35))
    etree.SubElement(ide, NFE+"cNF").text = sale.get("cNF","12345678")
    etree.SubElement(ide, NFE+"natOp").text = "VENDA"
    etree.SubElement(ide, NFE+"mod").text = "65"
    etree.SubElement(ide, NFE+"serie").text = str(cfg.serie)
    etree.SubElement(ide, NFE+"nNF").text = str(cfg.numero)
    etree.SubElement(ide, NFE+"dhEmi").text = datetime.datetime.now().isoformat()
    etree.SubElement(ide, NFE+"tpNF").text = "1"; etree.SubElement(ide, NFE+"idDest").text = "1"
    etree.SubElement(ide, NFE+"cMunFG").text = sale.get("cMunFG","3550308")
    etree.SubElement(ide, NFE+"tpImp").text = "4"; etree.SubElement(ide, NFE+"tpEmis").text="1"
    etree.SubElement(ide, NFE+"cDV").text = chave[-1]
    etree.SubElement(ide, NFE+"tpAmb").text = "2" if cfg.ambiente=="homolog" else "1"
    etree.SubElement(ide, NFE+"finNFe").text="1"; etree.SubElement(ide, NFE+"indFinal").text="1"
    etree.SubElement(ide, NFE+"indPres").text="1"; etree.SubElement(ide, NFE+"procEmi").text="0"
    etree.SubElement(ide, NFE+"verProc").text="ERP/PDV 1.0"

    emit = etree.SubElement(inf, NFE+"emit")
    etree.SubElement(emit, NFE+"CNPJ").text = only_digits(cfg.cnpj)
    etree.SubElement(emit, NFE+"xNome").text = sale.get("emit_nome","SUA EMPRESA")
    etree.SubElement(emit, NFE+"IE").text   = only_digits(cfg.ie)
    etree.SubElement(emit, NFE+"CRT").text  = "1" if (cfg.regime or "").lower().startswith("simples") else "3"
    ender = etree.SubElement(emit, NFE+"enderEmit")
    etree.SubElement(ender, NFE+"xLgr").text=sale.get("emit_ender","Rua X")
    etree.SubElement(ender, NFE+"nro").text =sale.get("emit_num","123")
    etree.SubElement(ender, NFE+"xBairro").text=sale.get("emit_bairro","Centro")
    etree.SubElement(ender, NFE+"cMun").text=sale.get("cMunFG","3550308")
    etree.SubElement(ender, NFE+"xMun").text=sale.get("emit_mun","São Paulo")
    etree.SubElement(ender, NFE+"UF").text  =cfg.uf
    etree.SubElement(ender, NFE+"CEP").text =sale.get("emit_cep","01000000")

    total_vprod = 0.0
    for idx,it in enumerate(items,1):
        det = etree.SubElement(inf, NFE+"det", nItem=str(idx)); prod = etree.SubElement(det, NFE+"prod")
        q = float(it.get("qty",1)); p = float(it.get("price",0)); vprod = q*p; total_vprod += vprod
        etree.SubElement(prod,NFE+"cProd").text=str(it.get("code",""))
        etree.SubElement(prod,NFE+"cEAN").text="SEM GTIN"
        etree.SubElement(prod,NFE+"xProd").text=it.get("desc","ITEM")
        etree.SubElement(prod,NFE+"NCM").text =it.get("ncm","00000000")
        etree.SubElement(prod,NFE+"CFOP").text=it.get("cfop","5102")
        etree.SubElement(prod,NFE+"uCom").text=it.get("unit","UN")
        etree.SubElement(prod,NFE+"qCom").text=f"{q:.4f}"
        etree.SubElement(prod,NFE+"vUnCom").text=f"{p:.4f}"
        etree.SubElement(prod,NFE+"vProd").text=f"{vprod:.2f}"
        etree.SubElement(prod,NFE+"cEANTrib").text="SEM GTIN"
        etree.SubElement(prod,NFE+"uTrib").text=it.get("unit","UN")
        etree.SubElement(prod,NFE+"qTrib").text=f"{q:.4f}"
        etree.SubElement(prod,NFE+"vUnTrib").text=f"{p:.4f}"
        etree.SubElement(prod,NFE+"indTot").text="1"

        imposto = etree.SubElement(det, NFE+"imposto")
        icms = etree.SubElement(imposto, NFE+"ICMS")
        icms40 = etree.SubElement(icms, NFE+"ICMS40")
        etree.SubElement(icms40, NFE+"orig").text="0"; etree.SubElement(icms40, NFE+"CST").text="40"
        pis = etree.SubElement(imposto, NFE+"PIS"); pisnt = etree.SubElement(pis, NFE+"PISNT")
        etree.SubElement(pisnt, NFE+"CST").text="07"
        cof = etree.SubElement(imposto, NFE+"COFINS"); cofnt = etree.SubElement(cof, NFE+"COFINSNT")
        etree.SubElement(cofnt, NFE+"CST").text="07"

    total = etree.SubElement(inf, NFE+"total"); icmstot = etree.SubElement(total, NFE+"ICMSTot")
    vs = {"vBC":"0.00","vICMS":"0.00","vICMSDeson":"0.00","vFCP":"0.00","vBCST":"0.00","vST":"0.00",
          "vFCPST":"0.00","vFCPSTRet":"0.00","vProd":f"{total_vprod:.2f}","vFrete":"0.00","vSeg":"0.00",
          "vDesc":"0.00","vII":"0.00","vIPI":"0.00","vIPIDevol":"0.00","vPIS":"0.00","vCOFINS":"0.00","vOutro":"0.00",
          "vNF":f"{total_vprod:.2f}"}
    for k,v in vs.items(): etree.SubElement(icmstot, NFE+k).text=v

    pag = etree.SubElement(inf, NFE+"pag"); detp = etree.SubElement(pag, NFE+"detPag")
    etree.SubElement(detp, NFE+"tPag").text = sale.get("tPag","01")  # 01 dinheiro, 03 cartão, 17 PIX
    etree.SubElement(detp, NFE+"vPag").text = f"{total_vprod:.2f}"

    info = etree.SubElement(inf, NFE+"infAdic"); etree.SubElement(info, NFE+"infCpl").text = f"DANFE NFC-e ambiente {cfg.ambiente}"
    infSupl = etree.SubElement(nfe, NFE+"infNFeSupl"); etree.SubElement(infSupl, NFE+"qrCode").text = qrcode_text(chave, cfg.csc_id, cfg.csc_token, total_vprod, cfg.uf, cfg.ambiente)
    return nfe

def sign_xml(nfe_xml: etree._Element, pfx: bytes, password: str) -> bytes:
    key_pem, cert_pem = _pfx_to_keycert(pfx, password)
    inf = nfe_xml.find(".//{http://www.portalfiscal.inf.br/nfe}infNFe")
    if inf is None: raise RuntimeError("infNFe não encontrado para assinatura.")
    signer = XMLSigner(method=methods.enveloped, signature_algorithm="rsa-sha1", digest_algorithm="sha1")
    signed = signer.sign(inf, key=key_pem, cert=cert_pem)
    parent = nfe_xml
    oldsig = parent.find(".//{http://www.w3.org/2000/09/xmldsig#}Signature")
    if oldsig is not None: parent.remove(oldsig)
    parent.append(signed)
    return etree.tostring(parent, encoding="utf-8", xml_declaration=True)

def gerar_pdf_danfe(xml_bytes: bytes, chave: str, out_pdf: pathlib.Path):
    out_pdf.parent.mkdir(parents=True, exist_ok=True)
    c = canvas.Canvas(str(out_pdf), pagesize=A4); w,h=A4
    c.setFont("Helvetica-Bold", 14); c.drawString(40, h-50, "DANFE NFC-e (modelo 65)")
    c.setFont("Helvetica", 10); c.drawString(40, h-70, "Chave: {0}".format(chave))
    qr = qrcode.make("https://consultar.nfce/{0}".format(chave)); buf = io.BytesIO(); qr.save(buf, format="PNG"); buf.seek(0)
    c.drawImage(ImageReader(buf), w-180, h-160, width=140, height=140, preserveAspectRatio=True, anchor='nw')
    c.drawString(40, h-90, "Documento gerado para conferência ({0}).".format(datetime.datetime.now().isoformat()))
    c.showPage(); c.save()

def abrir_pdf(path: str):
    sys = platform.system()
    try:
        if sys == "Windows":  os.startfile(path)             # abre no visualizador padrão
        elif sys == "Darwin": subprocess.run(["open", path], check=False)
        else:                 subprocess.run(["xdg-open", path], check=False)
    except Exception: pass

class NFCeAdapter:
    """Gera XML assinado + DANFE PDF e registra em invoices. Transmissão SEFAZ pode ser plugada depois."""
    def __init__(self, get_conn):
        self.get_conn = get_conn

    def emit(self, sale_id: int) -> Dict[str,Any]:
        cfg = load_config(self.get_conn)
        with self.get_conn() as con:
            con.row_factory = sqlite3.Row
            sale = con.execute("SELECT * FROM sales WHERE id=?", (sale_id,)).fetchone()
            if not sale: raise RuntimeError("Venda {0} não encontrada".format(sale_id))
            rows = list(con.execute("SELECT * FROM sale_items WHERE sale_id=? ORDER BY id", (sale_id,)))
        itens = []
        for r in rows:
            # tenta mapear campos comuns; adapta ao seu schema se necessário
            desc = r["description"] if "description" in r.keys() else str(r["product_id"])
            price = r["unit_price"] if "unit_price" in r.keys() else r["price"]
            qty = r["qty"]
            itens.append({"code": r["product_id"], "desc": desc, "ncm":"00000000", "cfop":"5102",
                          "unit":"UN", "qty": float(qty), "price": float(price)})

        agora = datetime.datetime.now(); cNF = f"{sale_id%99999999:08d}"
        chave = chave_acesso(cfg.uf, cfg.cnpj, agora, cfg.serie, cfg.numero, 1, cNF)
        sale_stub = {"tPag": "01", "emit_nome": "SUA EMPRESA"}  # ajuste se quiser
        nfe_xml = build_xml_nfce(sale_stub, itens, cfg, chave)
        xml_assinado = sign_xml(nfe_xml, cfg.cert_pfx, cfg.cert_password)

        # mock de autorização em HOMOLOG
        autorizado = (cfg.ambiente == "homolog"); status = "authorized" if autorizado else "pending"
        protocolo = "HOMOLOG-OK-{0}".format(chave[-6:]) if autorizado else None

        out_pdf = pathlib.Path.cwd()/ "nfce" / ("DANFE_NFCE_{0}.pdf".format(chave))
        gerar_pdf_danfe(xml_assinado, chave, out_pdf)

        with self.get_conn() as con:
            with _tx(con):
                cur = con.execute(
                  "INSERT INTO invoices(sale_id, model, chave, numero, serie, status, protocolo, xml, pdf_path, created_at, authorized_at) "
                  "VALUES(?,?,?,?,?,?,?,?,?,?,?)",
                  (sale_id, "NFCE", chave, cfg.numero, cfg.serie, status, protocolo, xml_assinado.decode("utf-8"),
                   str(out_pdf), datetime.datetime.now().isoformat(), datetime.datetime.now().isoformat() if autorizado else None)
                )
                inv_id = cur.lastrowid
                for it in itens:
                    con.execute(
                      "INSERT INTO invoice_items(invoice_id, product_id, description, ncm, cfop, qty, unit, unit_price, total, tax_json) "
                      "VALUES(?,?,?,?,?,?,?,?,?,?)",
                      (inv_id, it["code"], it["desc"], it["ncm"], it["cfop"], it["qty"], it["unit"], it["price"],
                       it["qty"]*it["price"], json.dumps({"icms":"40","pis":"07","cofins":"07"}))
                    )
                con.execute("UPDATE fiscal_config SET numero = numero + 1, updated_at=datetime('now') WHERE id=1")

        return {"ok": True, "chave": chave, "pdf": str(out_pdf), "status": status, "protocolo": protocolo}
