# online_guard.py
from __future__ import annotations
import sqlite3, datetime, email.utils, ssl
from typing import Optional, Dict
try:
    import requests
    _REQ = True
except Exception:
    _REQ = False

TRUST_HOSTS = [
    "https://www.google.com",
    "https://www.cloudflare.com",
    "https://www.microsoft.com",
]
OFFLINE_GRACE_HOURS = 24  # janela offline

def _get_setting(con: sqlite3.Connection, key: str, default: str = "") -> str:
    con.execute("CREATE TABLE IF NOT EXISTS app_state (key TEXT PRIMARY KEY, value TEXT)")
    row = con.execute("SELECT value FROM app_state WHERE key=?", (key,)).fetchone()
    return row[0] if row and row[0] is not None else default

def _set_setting(con: sqlite3.Connection, key: str, value: str) -> None:
    con.execute("CREATE TABLE IF NOT EXISTS app_state (key TEXT PRIMARY KEY, value TEXT)")
    con.execute("""INSERT INTO app_state(key,value) VALUES(?,?)
                   ON CONFLICT(key) DO UPDATE SET value=excluded.value""", (key, value))

def get_trusted_time(timeout: int = 6) -> Optional[datetime.datetime]:
    """Retorna datetime UTC baseado no header Date de hosts confiáveis."""
    if not _REQ:
        return None
    for url in TRUST_HOSTS:
        try:
            r = requests.head(url, timeout=timeout)
            s = r.headers.get("Date") or r.headers.get("date")
            if s:
                dt = email.utils.parsedate_to_datetime(s)
                if dt.tzinfo:
                    return dt.astimezone(datetime.timezone.utc).replace(tzinfo=None)
                return dt.replace(tzinfo=None)
        except Exception:
            continue
    return None

def evaluate_online_mode(con: sqlite3.Connection) -> Dict[str, bool]:
    """
    Retorna:
      - online: True/False
      - offline_allowed: True se dentro da janela de 24h desde a última validação
    Atualiza last_online_at quando conseguir hora confiável.
    """
    now_utc = datetime.datetime.utcnow()
    trusted = get_trusted_time()
    if trusted:
        _set_setting(con, "last_online_at", trusted.isoformat())
        return {"online": True, "offline_allowed": True}

    last = _get_setting(con, "last_online_at", "")
    if not last:
        # nunca validou online
        return {"online": False, "offline_allowed": False}

    try:
        last_dt = datetime.datetime.fromisoformat(last)
    except Exception:
        return {"online": False, "offline_allowed": False}

    delta = now_utc - last_dt
    if delta.total_seconds() <= OFFLINE_GRACE_HOURS * 3600:
        return {"online": False, "offline_allowed": True}
    return {"online": False, "offline_allowed": False}
