# payment_service.py
import os
import time
import json
import uuid
import qrcode
import sqlite3
from datetime import datetime
from typing import Optional, Tuple, Dict, Any
import requests

DEFAULT_TIMEOUT_SEC = 180  # 3 min
POLL_INTERVAL_SEC = 2

class PaymentService:
    """
    Serviço de pagamento PIX (Mercado Pago) com geração de QR e polling de status.
    Depende das chaves no settings: MP_ACCESS_TOKEN, MP_NOTIFICATION_URL (opcional)
    Tabela: payments (mapeia sale_id -> status/txid/external_id)
    """

    def __init__(self, db_path: str, media_dir: str = "media"):
        self.db_path = db_path
        self.media_dir = media_dir
        os.makedirs(self.media_dir, exist_ok=True)

    # ---------- Utils DB ----------
    def _conn(self):
        conn = sqlite3.connect(self.db_path)
        conn.row_factory = sqlite3.Row
        return conn

    def _now(self) -> str:
        return datetime.utcnow().strftime("%Y-%m-%d %H:%M:%S")

    def _get_setting(self, key: str) -> Optional[str]:
        with self._conn() as cx:
            row = cx.execute("SELECT value FROM settings WHERE key=?", (key,)).fetchone()
            return row["value"] if row else None

    # ---------- Mercado Pago ----------
    def _mp_headers(self) -> Dict[str, str]:
        token = self._get_setting("MP_ACCESS_TOKEN")
        if not token:
            raise RuntimeError("Configuração ausente: MP_ACCESS_TOKEN")
        return {
            "Authorization": f"Bearer {token}",
            "Content-Type": "application/json",
            "X-Idempotency-Key": str(uuid.uuid4())
        }

    def create_pix(self, sale_id: int, amount: float, description: str) -> Dict[str, Any]:
        """Cria cobrança PIX no MP e registra em payments."""
        external_ref = f"sale-{sale_id}-{uuid.uuid4().hex[:8]}"
        body = {
            "transaction_amount": float(round(amount, 2)),
            "description": description[:250],
            "payment_method_id": "pix",
            "payer": {"email": "comprador@exemplo.com"},  # opcional
            "external_reference": external_ref
        }
        notif = self._get_setting("MP_NOTIFICATION_URL")
        if notif:
            body["notification_url"] = notif

        resp = requests.post(
            "https://api.mercadopago.com/v1/payments",
            headers=self._mp_headers(),
            data=json.dumps(body),
            timeout=20
        )
        if resp.status_code not in (200, 201):
            raise RuntimeError(f"Falha ao criar cobrança PIX: {resp.status_code} {resp.text}")

        data = resp.json()
        qr_text = data.get("point_of_interaction", {}).get("transaction_data", {}).get("qr_code")
        txid = data.get("id")
        if not qr_text or not txid:
            raise RuntimeError("Resposta MP sem qr_code/txid")

        # Gera PNG do QR
        png_path = os.path.join(self.media_dir, f"pix_{sale_id}_{txid}.png")
        img = qrcode.make(qr_text)
        img.save(png_path)

        # Persiste payment
        with self._conn() as cx:
            cx.execute("""
                INSERT INTO payments (sale_id, provider, qr_text, qr_image_path, external_id, txid, status, created_at, updated_at)
                VALUES (?,?,?,?,?,?, 'PENDING', ?, ?)
            """, (sale_id, "mercadopago", qr_text, png_path, external_ref, str(txid), self._now(), self._now()))
        return {"qr_text": qr_text, "qr_image_path": png_path, "external_id": external_ref, "txid": str(txid)}

    def poll_until_paid(self, txid: str, timeout_sec: int = DEFAULT_TIMEOUT_SEC) -> str:
        """
        Faz polling no MP até status final. Retorna status final (PAID|CANCELLED|EXPIRED|ERROR|PENDING).
        """
        start = time.time()
        status_map = {
            "approved": "PAID",
            "rejected": "CANCELLED",
            "cancelled": "CANCELLED",
            "refunded": "CANCELLED",
            "charged_back": "CANCELLED",
            "expired": "EXPIRED",
            "in_process": "PENDING",
            "pending": "PENDING"
        }
        while time.time() - start <= timeout_sec:
            r = requests.get(
                f"https://api.mercadopago.com/v1/payments/{txid}",
                headers=self._mp_headers(),
                timeout=15
            )
            if r.status_code != 200:
                time.sleep(POLL_INTERVAL_SEC)
                continue
            st = r.json().get("status", "pending")
            final = status_map.get(st, "PENDING")
            if final in ("PAID", "CANCELLED", "EXPIRED"):
                return final
            time.sleep(POLL_INTERVAL_SEC)
        return "PENDING"

    # ---------- Integração vendas ----------
    def mark_payment_status(self, sale_id: int, txid: str, new_status: str):
        with self._conn() as cx:
            cx.execute("UPDATE payments SET status=?, updated_at=? WHERE txid=?",
                       (new_status, self._now(), txid))
            if new_status == "PAID":
                cx.execute("UPDATE sales SET status='PAGA' WHERE id=?", (sale_id,))
            elif new_status in ("CANCELLED", "EXPIRED"):
                cx.execute("UPDATE sales SET status='CANCELADA' WHERE id=?", (sale_id,))

    def ensure_payment_record(self, sale_id: int) -> Optional[sqlite3.Row]:
        with self._conn() as cx:
            row = cx.execute("SELECT * FROM payments WHERE sale_id=? ORDER BY id DESC LIMIT 1", (sale_id,)).fetchone()
            return row
