# pix_config.py — Dialogo de Configuração do PIX
from __future__ import annotations
import json
import sqlite3
import tkinter as tk
from tkinter import ttk, messagebox
from typing import Callable, Optional

# requests é opcional; só usamos no teste
try:
    import requests
    _REQ_OK = True
except Exception:
    _REQ_OK = False

# ----------------------------------------
# Helpers DB
# ----------------------------------------
def _ensure_settings(conn: sqlite3.Connection):
    conn.execute("""
        CREATE TABLE IF NOT EXISTS settings (
            key   TEXT PRIMARY KEY,
            value TEXT
        )
    """)

def _get(conn: sqlite3.Connection, key: str, default: str = "") -> str:
    _ensure_settings(conn)
    r = conn.execute("SELECT value FROM settings WHERE key=?", (key,)).fetchone()
    return r[0] if r and r[0] is not None else default

def _set(conn: sqlite3.Connection, key: str, value: str):
    _ensure_settings(conn)
    conn.execute("""
        INSERT INTO settings(key,value) VALUES(?,?)
        ON CONFLICT(key) DO UPDATE SET value=excluded.value
    """, (key, value if value is not None else ""))

# ----------------------------------------
# Dialog
# ----------------------------------------
class PixConfigDialog:
    def __init__(self, master: tk.Misc, get_conn: Callable[[], sqlite3.Connection]):
        self.master = master
        self.get_conn = get_conn

        self.win = tk.Toplevel(master)
        self.win.title("Configuração PIX")
        self.win.transient(master)
        self.win.grab_set()
        self.win.geometry("760x620")
        self.win.minsize(740, 560)

        self._build_vars()
        self._build_ui()
        self._load_from_db()
        self._toggle_provider_sections()

    # ---------------- UI -----------------
    def _build_vars(self):
        self.var_provider  = tk.StringVar(value="mercadopago")
        self.var_key       = tk.StringVar(value="")  # QR estático
        self.var_api_base  = tk.StringVar(value="https://api.mercadopago.com")
        self.var_client_id = tk.StringVar(value="")
        self.var_token     = tk.StringVar(value="")
        self.var_poll      = tk.StringVar(value="3")
        self.var_txid_pref = tk.StringVar(value="FD")
        self.var_email     = tk.StringVar(value="cliente@example.com")

        # Genérico (HTTP)
        self.var_g_create_url     = tk.StringVar(value="")
        self.var_g_create_method  = tk.StringVar(value="POST")
        self.var_g_create_headers = tk.StringVar(value="")
        self.var_g_create_body    = tk.StringVar(value="")
        self.var_g_qr_path        = tk.StringVar(value="")
        self.var_g_id_path        = tk.StringVar(value="")
        self.var_g_status_url     = tk.StringVar(value="")
        self.var_g_status_headers = tk.StringVar(value="")
        self.var_g_status_path    = tk.StringVar(value="")
        self.var_g_status_paid    = tk.StringVar(value="approved")

    def _build_ui(self):
        root = self.win
        root.columnconfigure(0, weight=1)

        # Provedor
        frm_top = ttk.LabelFrame(root, text="Provedor", padding=8)
        frm_top.grid(row=0, column=0, sticky="ew", padx=8, pady=(8,4))
        frm_top.columnconfigure(1, weight=1)

        ttk.Label(frm_top, text="Provedor (PSP):").grid(row=0, column=0, sticky="e", padx=(0,6))
        cmb = ttk.Combobox(frm_top, textvariable=self.var_provider, state="readonly",
                           values=["mercadopago", "simulado", "generico-http"], width=20)
        cmb.grid(row=0, column=1, sticky="w")
        cmb.bind("<<ComboboxSelected>>", lambda e: self._toggle_provider_sections())

        ttk.Label(frm_top, text="Chave PIX (opcional):").grid(row=1, column=0, sticky="e", padx=(0,6), pady=(6,0))
        ttk.Entry(frm_top, textvariable=self.var_key, width=50).grid(row=1, column=1, sticky="ew", pady=(6,0))

        # Credenciais
        frm_cred = ttk.LabelFrame(root, text="Credenciais", padding=8)
        frm_cred.grid(row=1, column=0, sticky="ew", padx=8, pady=4)
        for i in range(2): frm_cred.columnconfigure(i, weight=1)

        ttk.Label(frm_cred, text="API Base (opcional):").grid(row=0, column=0, sticky="e", padx=(0,6))
        ttk.Entry(frm_cred, textvariable=self.var_api_base).grid(row=0, column=1, sticky="ew")

        ttk.Label(frm_cred, text="Client ID (opcional):").grid(row=1, column=0, sticky="e", padx=(0,6), pady=(6,0))
        ttk.Entry(frm_cred, textvariable=self.var_client_id).grid(row=1, column=1, sticky="ew", pady=(6,0))

        ttk.Label(frm_cred, text="Access Token (APP_USR-...):").grid(row=2, column=0, sticky="e", padx=(0,6), pady=(6,0))
        ent_token = ttk.Entry(frm_cred, textvariable=self.var_token, show="*")
        ent_token.grid(row=2, column=1, sticky="ew", pady=(6,0))

        # Opções
        frm_opts = ttk.LabelFrame(root, text="Opções", padding=8)
        frm_opts.grid(row=2, column=0, sticky="ew", padx=8, pady=4)
        frm_opts.columnconfigure(1, weight=1)

        ttk.Label(frm_opts, text="Intervalo polling (s):").grid(row=0, column=0, sticky="e", padx=(0,6))
        ttk.Entry(frm_opts, textvariable=self.var_poll, width=10).grid(row=0, column=1, sticky="w")

        ttk.Label(frm_opts, text="Prefixo TXID (opcional):").grid(row=1, column=0, sticky="e", padx=(0,6), pady=(6,0))
        ttk.Entry(frm_opts, textvariable=self.var_txid_pref, width=14).grid(row=1, column=1, sticky="w", pady=(6,0))

        ttk.Label(frm_opts, text="E-mail do pagador (opcional):").grid(row=2, column=0, sticky="e", padx=(0,6), pady=(6,0))
        ttk.Entry(frm_opts, textvariable=self.var_email, width=30).grid(row=2, column=1, sticky="w", pady=(6,0))

        # Ações topo
        frm_actions = ttk.Frame(root)
        frm_actions.grid(row=3, column=0, sticky="ew", padx=8, pady=(0,4))
        ttk.Button(frm_actions, text="Salvar", command=self._save).pack(side="right", padx=(6,0))
        ttk.Button(frm_actions, text="Testar conexão (MP)", command=self._test_mp).pack(side="right")

        # Genérico HTTP
        self.frm_gen = ttk.LabelFrame(root, text="Genérico (HTTP) — somente quando selecionado", padding=8)
        self.frm_gen.grid(row=4, column=0, sticky="nsew", padx=8, pady=(4,8))
        self.frm_gen.columnconfigure(1, weight=1)
        root.rowconfigure(4, weight=1)

        r = 0
        ttk.Label(self.frm_gen, text="Create URL:").grid(row=r, column=0, sticky="e", padx=(0,6)); r+=1
        ttk.Entry(self.frm_gen, textvariable=self.var_g_create_url).grid(row=r-1, column=1, sticky="ew")

        ttk.Label(self.frm_gen, text="Create method:").grid(row=r, column=0, sticky="e", padx=(0,6)); r+=1
        ttk.Entry(self.frm_gen, textvariable=self.var_g_create_method).grid(row=r-1, column=1, sticky="w")

        ttk.Label(self.frm_gen, text="Create headers (JSON):").grid(row=r, column=0, sticky="ne", padx=(0,6)); r+=1
        tk.Text(self.frm_gen, name="g_headers", height=4).grid(row=r-1, column=1, sticky="nsew"); self.frm_gen.rowconfigure(r-1, weight=1)

        ttk.Label(self.frm_gen, text="Create body (JSON):").grid(row=r, column=0, sticky="ne", padx=(0,6)); r+=1
        tk.Text(self.frm_gen, name="g_body", height=5).grid(row=r-1, column=1, sticky="nsew")

        ttk.Label(self.frm_gen, text="QR path (dot):").grid(row=r, column=0, sticky="e", padx=(0,6)); r+=1
        ttk.Entry(self.frm_gen, textvariable=self.var_g_qr_path).grid(row=r-1, column=1, sticky="ew")

        ttk.Label(self.frm_gen, text="ID path (dot):").grid(row=r, column=0, sticky="e", padx=(0,6)); r+=1
        ttk.Entry(self.frm_gen, textvariable=self.var_g_id_path).grid(row=r-1, column=1, sticky="ew")

        ttk.Label(self.frm_gen, text="Status URL ({payment_id}):").grid(row=r, column=0, sticky="e", padx=(0,6)); r+=1
        ttk.Entry(self.frm_gen, textvariable=self.var_g_status_url).grid(row=r-1, column=1, sticky="ew")

        ttk.Label(self.frm_gen, text="Status headers (JSON):").grid(row=r, column=0, sticky="ne", padx=(0,6)); r+=1
        tk.Text(self.frm_gen, name="g_sheaders", height=3).grid(row=r-1, column=1, sticky="nsew")

        ttk.Label(self.frm_gen, text="Status path (dot):").grid(row=r, column=0, sticky="e", padx=(0,6)); r+=1
        ttk.Entry(self.frm_gen, textvariable=self.var_g_status_path).grid(row=r-1, column=1, sticky="ew")

        ttk.Label(self.frm_gen, text="Status ‘pago’ (valor):").grid(row=r, column=0, sticky="e", padx=(0,6)); r+=1
        ttk.Entry(self.frm_gen, textvariable=self.var_g_status_paid).grid(row=r-1, column=1, sticky="w")

        # Rodapé
        frm_bottom = ttk.Frame(root); frm_bottom.grid(row=5, column=0, sticky="ew", padx=8, pady=(0,8))
        ttk.Button(frm_bottom, text="Fechar", command=self.win.destroy).pack(side="right")

    # ---------------- DATA -----------------
    def _load_from_db(self):
        with self.get_conn() as conn:
            self.var_provider.set(_get(conn, "PIX_PROVIDER", "mercadopago") or "mercadopago")
            self.var_key.set(_get(conn, "PIX_KEY", ""))
            self.var_api_base.set(_get(conn, "PIX_API_BASE", "https://api.mercadopago.com") or "https://api.mercadopago.com")
            self.var_client_id.set(_get(conn, "PIX_CLIENT_ID", ""))
            self.var_token.set(_get(conn, "PIX_CLIENT_SECRET", ""))
            self.var_poll.set(_get(conn, "PIX_POLL_INTERVAL", "3") or "3")
            self.var_txid_pref.set(_get(conn, "PIX_TXID_PREFIX", "FD") or "FD")
            self.var_email.set(_get(conn, "PIX_PAYER_EMAIL", "cliente@example.com") or "cliente@example.com")

            # Genérico
            self.var_g_create_url.set(_get(conn, "GEN_CREATE_URL", ""))
            self.var_g_create_method.set(_get(conn, "GEN_CREATE_METHOD", "POST") or "POST")
            self.var_g_create_headers.set(_get(conn, "GEN_CREATE_HEADERS", ""))
            self.var_g_create_body.set(_get(conn, "GEN_CREATE_BODY", ""))
            self.var_g_qr_path.set(_get(conn, "GEN_CREATE_QR_PATH", ""))
            self.var_g_id_path.set(_get(conn, "GEN_CREATE_ID_PATH", ""))
            self.var_g_status_url.set(_get(conn, "GEN_STATUS_URL", ""))
            self.var_g_status_headers.set(_get(conn, "GEN_STATUS_HEADERS", ""))
            self.var_g_status_path.set(_get(conn, "GEN_STATUS_PATH", ""))
            self.var_g_status_paid.set(_get(conn, "GEN_STATUS_PAID", "approved") or "approved")

        # Carrega textos nos Text widgets do Genérico
        try:
            self.frm_gen.nametowidget("g_headers").delete("1.0", "end")
            self.frm_gen.nametowidget("g_headers").insert("1.0", self.var_g_create_headers.get())
            self.frm_gen.nametowidget("g_body").delete("1.0", "end")
            self.frm_gen.nametowidget("g_body").insert("1.0", self.var_g_create_body.get())
            self.frm_gen.nametowidget("g_sheaders").delete("1.0", "end")
            self.frm_gen.nametowidget("g_sheaders").insert("1.0", self.var_g_status_headers.get())
        except Exception:
            pass

    def _save(self):
        # validação básica
        prov = self.var_provider.get().strip().lower()
        poll = self.var_poll.get().strip()
        try:
            poll_i = max(2, int(poll or "3"))
        except Exception:
            poll_i = 3

        if prov == "mercadopago":
            if not self.var_token.get().strip():
                if not messagebox.askyesno("PIX", "Access Token está vazio. Salvar mesmo assim?", parent=self.win):
                    return

        # Persistir
        with self.get_conn() as conn:
            _set(conn, "PIX_PROVIDER", prov)
            _set(conn, "PIX_KEY", self.var_key.get().strip())
            _set(conn, "PIX_API_BASE", self.var_api_base.get().strip())
            _set(conn, "PIX_CLIENT_ID", self.var_client_id.get().strip())
            _set(conn, "PIX_CLIENT_SECRET", self.var_token.get().strip())
            _set(conn, "PIX_POLL_INTERVAL", str(poll_i))
            _set(conn, "PIX_TXID_PREFIX", self.var_txid_pref.get().strip() or "FD")
            _set(conn, "PIX_PAYER_EMAIL", self.var_email.get().strip())

            # Modo de confirmação padrão quando provedor = mercadopago
            if prov == "mercadopago":
                _set(conn, "PIX_CONFIRM_MODE", "polling")
            elif prov == "simulado":
                _set(conn, "PIX_CONFIRM_MODE", "none")
            else:
                _set(conn, "PIX_CONFIRM_MODE", "auto")  # genérico: deixa livre

            # Genérico (HTTP)
            # captura textos dos widgets
            try:
                self.var_g_create_headers.set(self.frm_gen.nametowidget("g_headers").get("1.0", "end").strip())
                self.var_g_create_body.set(self.frm_gen.nametowidget("g_body").get("1.0", "end").strip())
                self.var_g_status_headers.set(self.frm_gen.nametowidget("g_sheaders").get("1.0", "end").strip())
            except Exception:
                pass

            _set(conn, "GEN_CREATE_URL", self.var_g_create_url.get().strip())
            _set(conn, "GEN_CREATE_METHOD", self.var_g_create_method.get().strip() or "POST")
            _set(conn, "GEN_CREATE_HEADERS", self.var_g_create_headers.get())
            _set(conn, "GEN_CREATE_BODY", self.var_g_create_body.get())
            _set(conn, "GEN_CREATE_QR_PATH", self.var_g_qr_path.get().strip())
            _set(conn, "GEN_CREATE_ID_PATH", self.var_g_id_path.get().strip())
            _set(conn, "GEN_STATUS_URL", self.var_g_status_url.get().strip())
            _set(conn, "GEN_STATUS_HEADERS", self.var_g_status_headers.get())
            _set(conn, "GEN_STATUS_PATH", self.var_g_status_path.get().strip())
            _set(conn, "GEN_STATUS_PAID", self.var_g_status_paid.get().strip() or "approved")

            conn.commit()

        messagebox.showinfo("PIX", "Configurações salvas.", parent=self.win)

    def _toggle_provider_sections(self):
        prov = (self.var_provider.get() or "").strip().lower()
        # Genérico visível só quando 'generico-http'
        if prov == "generico-http":
            self.frm_gen.grid()  # mostra
        else:
            self.frm_gen.grid_remove()

    def _test_mp(self):
        if not _REQ_OK:
            messagebox.showerror("PIX", "Instale a dependência: pip install requests", parent=self.win)
            return
        token = self.var_token.get().strip()
        if not token:
            messagebox.showwarning("PIX", "Informe o Access Token para testar.", parent=self.win)
            return
        base = self.var_api_base.get().strip() or "https://api.mercadopago.com"
        url = base.rstrip("/") + "/v1/payments/search?sort=date_created&criteria=desc&limit=1"
        try:
            r = requests.get(url, headers={"Authorization": f"Bearer {token}"}, timeout=12)
            if r.status_code == 200:
                messagebox.showinfo("PIX", "Conectado com sucesso à API Mercado Pago.", parent=self.win)
            elif r.status_code == 401:
                messagebox.showerror("PIX", "Token inválido (401). Verifique se é APP_USR- e as permissões.", parent=self.win)
            else:
                messagebox.showerror("PIX", f"Falha ({r.status_code}): {r.text[:300]}", parent=self.win)
        except Exception as e:
            messagebox.showerror("PIX", f"Erro de rede:\n{e}", parent=self.win)

# API pública
def open_pix_config(app, get_conn: Callable[[], sqlite3.Connection]):
    return PixConfigDialog(app.root, get_conn)
