# pix_providers.py — abstração de PSPs de PIX dinâmico com polling
from __future__ import annotations
import os
import json
from typing import Any, Dict, Tuple

try:
    import requests
except Exception:  # pragma: no cover
    requests = None  # avisaremos na criação


class ProviderError(RuntimeError):
    pass


def _fingerprint(tok: str) -> str:
    if not tok:
        return "(vazio)"
    return f"{tok[:12]}…{tok[-4:]}"


# ---------- Provedor base ----------
class BaseProvider:
    def __init__(self, settings: Dict[str, Any]):
        self.s = settings

    def create_payment(self, amount: float, description: str, payer_email: str = "") -> Tuple[str, str]:
        """
        Retorna (qr_code_text, payment_id)
        """
        raise NotImplementedError

    def check_paid(self, payment_id: str) -> bool:
        """
        True se pago (approved/accredited/etc)
        """
        raise NotImplementedError


# ---------- Mercado Pago ----------
class MercadoPagoProvider(BaseProvider):
    API_BASE_KEY = "PIX_API_BASE"
    TOKEN_KEY = "PIX_ACCESS_TOKEN"
    PAYER_EMAIL_KEY = "PIX_PAYER_EMAIL"

    def _require_requests(self):
        if requests is None:
            raise ProviderError("Biblioteca 'requests' não instalada. Faça: pip install requests")

    def _api(self, path: str) -> str:
        base = (self.s.get(self.API_BASE_KEY) or "https://api.mercadopago.com").rstrip("/")
        return f"{base}{path}"

    def _token(self) -> str:
        tok = (self.s.get(self.TOKEN_KEY) or "").strip()
        if not tok:
            raise ProviderError("Access Token de PRODUÇÃO não configurado (APP_USR-…).")
        return tok

    def _headers(self) -> Dict[str, str]:
        return {
            "Authorization": f"Bearer {self._token()}",
            "Content-Type": "application/json",
        }

    def _validate_token(self):
        url = self._api("/users/me")
        r = requests.get(url, headers=self._headers(), timeout=15)
        if r.status_code != 200:
            msg = ""
            try:
                j = r.json()
                msg = j.get("message") or j.get("error") or r.text
            except Exception:
                msg = r.text
            raise ProviderError(f"Token inválido ({r.status_code}): {msg} — {_fingerprint(self._token())}")

    def create_payment(self, amount: float, description: str, payer_email: str = "") -> Tuple[str, str]:
        self._require_requests()
        self._validate_token()

        url = self._api("/v1/payments")
        body = {
            "transaction_amount": float(amount),
            "description": description,
            "payment_method_id": "pix",
            "external_reference": self.s.get("PIX_TX_PREFIX", "FD") + "_PDV",
            "payer": {"email": payer_email or self.s.get(self.PAYER_EMAIL_KEY) or "cliente@example.com"},
        }

        # idempotência
        import random, string
        idem = self.s.get("PIX_TX_PREFIX", "FD") + "".join(random.choice(string.ascii_uppercase + string.digits) for _ in range(10))
        headers = dict(self._headers())
        headers["X-Idempotency-Key"] = idem

        r = requests.post(url, json=body, headers=headers, timeout=20)
        if r.status_code in (400, 409) and "Idempotency" in (r.text or ""):
            headers["X-Idempotency-Key"] = idem + "2"
            r = requests.post(url, json=body, headers=headers, timeout=20)

        if r.status_code >= 400:
            try:
                j = r.json(); msg = j.get("message") or j.get("error") or r.text
            except Exception:
                msg = r.text
            raise ProviderError(f"Erro {r.status_code} ao criar cobrança: {msg}")

        j = r.json()
        tdata = j.get("point_of_interaction", {}).get("transaction_data", {})
        qr = tdata.get("qr_code")
        pid = str(j.get("id") or "")
        if not qr or not pid:
            raise ProviderError("Resposta sem qr_code/payment_id.")
        return qr, pid

    def check_paid(self, payment_id: str) -> bool:
        self._require_requests()
        url = self._api(f"/v1/payments/{payment_id}")
        r = requests.get(url, headers=self._headers(), timeout=15)
        if r.status_code >= 400:
            # silencioso — tratamos como não pago
            return False
        j = r.json()
        st = (j.get("status") or "").lower()
        sd = (j.get("status_detail") or "").lower()
        return st in ("approved", "accredited", "authorized") or ("accredited" in sd)


# ---------- Genérico HTTP (template) ----------
class GenericHTTPProvider(BaseProvider):
    """
    Permite integrar QUALQUER PSP por configuração:
    - GEN_CREATE_URL (str)
    - GEN_CREATE_METHOD (str) [POST/GET] (padrão: POST)
    - GEN_CREATE_HEADERS (json dict, pode usar {access_token})
    - GEN_CREATE_BODY (json dict, pode usar {amount}, {description}, {txid}, {payer_email}, {access_token})
    - GEN_CREATE_QR_PATH (dot path, ex.: "data.qr_code")
    - GEN_CREATE_ID_PATH (dot path)
    - GEN_STATUS_URL (str; pode conter {payment_id})
    - GEN_STATUS_HEADERS (json dict, pode usar {access_token})
    - GEN_STATUS_PATH (dot path do status, ex.: "data.status")
    - GEN_STATUS_PAID (lista json/str com valores que significam 'pago', ex.: ["approved","paid","accredited"])
    """

    def _require_requests(self):
        if requests is None:
            raise ProviderError("Biblioteca 'requests' não instalada. Faça: pip install requests")

    # -------- dot path helpers --------
    @staticmethod
    def _get_by_path(obj: Any, path: str) -> Any:
        if not path:
            return None
        cur = obj
        for p in path.split("."):
            if isinstance(cur, dict) and p in cur:
                cur = cur[p]
            else:
                return None
        return cur

    @staticmethod
    def _fmt(value: str, **kwargs) -> str:
        return (value or "").format(**kwargs)

    def _json_or_dict(self, value: str, default) -> Any:
        if not value:
            return default
        try:
            return json.loads(value)
        except Exception:
            return default

    def create_payment(self, amount: float, description: str, payer_email: str = "") -> Tuple[str, str]:
        self._require_requests()

        access_token = (self.s.get("PIX_ACCESS_TOKEN") or "").strip()
        create_url = self.s.get("GEN_CREATE_URL") or ""
        method = (self.s.get("GEN_CREATE_METHOD") or "POST").upper()
        headers = self._json_or_dict(self.s.get("GEN_CREATE_HEADERS") or "", {})
        body = self._json_or_dict(self.s.get("GEN_CREATE_BODY") or "", {})

        # substituições básicas
        import random, string
        txid = self.s.get("PIX_TX_PREFIX", "FD") + "".join(random.choice(string.ascii_uppercase + string.digits) for _ in range(10))

        # format headers/body
        headers = {k: self._fmt(str(v), access_token=access_token) for k, v in headers.items()}
        def _walk_fmt(x):
            if isinstance(x, dict):
                return {k: _walk_fmt(v) for k, v in x.items()}
            if isinstance(x, list):
                return [_walk_fmt(v) for v in x]
            if isinstance(x, str):
                return self._fmt(x, amount=amount, description=description, payer_email=payer_email,
                                 access_token=access_token, txid=txid)
            return x
        body = _walk_fmt(body)

        if not create_url:
            raise ProviderError("GEN_CREATE_URL não configurado (vá em Configuração PIX > Avançado do provedor genérico).")

        # requisição
        try:
            if method == "GET":
                r = requests.get(create_url, headers=headers, timeout=20)
            else:
                r = requests.post(create_url, json=body, headers=headers, timeout=20)
        except Exception as e:
            raise ProviderError(f"Falha HTTP ao criar cobrança: {e}")

        if r.status_code >= 400:
            raise ProviderError(f"Erro {r.status_code} ao criar cobrança: {r.text[:300]}")

        try:
            j = r.json()
        except Exception:
            raise ProviderError("Resposta não é JSON.")

        qr_path = self.s.get("GEN_CREATE_QR_PATH") or ""
        id_path = self.s.get("GEN_CREATE_ID_PATH") or ""
        qr = self._get_by_path(j, qr_path)
        pid = self._get_by_path(j, id_path)
        if not qr or not pid:
            raise ProviderError(f"Falha ao extrair QR/ID. Paths usados: qr='{qr_path}' id='{id_path}'.")
        return str(qr), str(pid)

    def check_paid(self, payment_id: str) -> bool:
        self._require_requests()
        status_url = self.s.get("GEN_STATUS_URL") or ""
        if not status_url:
            return False
        access_token = (self.s.get("PIX_ACCESS_TOKEN") or "").strip()
        status_url = self._fmt(status_url, payment_id=payment_id, access_token=access_token)

        headers = self._json_or_dict(self.s.get("GEN_STATUS_HEADERS") or "", {})
        headers = {k: self._fmt(str(v), access_token=access_token, payment_id=payment_id) for k, v in headers.items()}
        try:
            r = requests.get(status_url, headers=headers, timeout=15)
        except Exception:
            return False
        if r.status_code >= 400:
            return False
        try:
            j = r.json()
        except Exception:
            return False
        path = self.s.get("GEN_STATUS_PATH") or ""
        paid_vals = self._json_or_dict(self.s.get("GEN_STATUS_PAID") or "", ["approved","paid","accredited"])
        status_val = self._get_by_path(j, path)
        if status_val is None:
            return False
        if isinstance(paid_vals, list):
            return str(status_val).lower() in [str(x).lower() for x in paid_vals]
        return str(status_val).lower() == str(paid_vals).lower()


# ---------- fábrica ----------
def get_provider(settings: Dict[str, Any]) -> BaseProvider:
    code = (settings.get("PIX_PROVIDER") or "mercadopago").lower()
    if code == "mercadopago":
        return MercadoPagoProvider(settings)
    if code in ("generic_http", "generico", "http"):
        return GenericHTTPProvider(settings)
    raise ProviderError(f"Provedor '{code}' não suportado. Use 'mercadopago' ou 'generic_http'.")
