# qr_dialog.py
import tkinter as tk
from tkinter import ttk, messagebox
from PIL import Image, ImageTk
import pyperclip
import os

QR_SIZE = 250

class QRDialog(tk.Toplevel):
    def __init__(self, parent, qr_image_path: str, qr_text: str, title: str = "Pague com PIX"):
        super().__init__(parent)
        self.title(title)
        self.resizable(False, False)
        self.grab_set()
        self.protocol("WM_DELETE_WINDOW", self._on_close)

        frm = ttk.Frame(self, padding=12)
        frm.pack(fill="both", expand=True)

        if not os.path.exists(qr_image_path):
            messagebox.showerror("Erro", "Imagem de QR não encontrada.")
            self.destroy()
            return

        img = Image.open(qr_image_path).resize((QR_SIZE, QR_SIZE))
        self.photo = ImageTk.PhotoImage(img)
        ttk.Label(frm, image=self.photo).pack(pady=(0, 8))

        self.qr_text = qr_text
        txt = tk.Text(frm, width=46, height=3, wrap="word")
        txt.insert("1.0", qr_text)
        txt.configure(state="disabled")
        txt.pack(pady=(0, 8))

        btns = ttk.Frame(frm)
        btns.pack()
        ttk.Button(btns, text="Copiar PIX (Copia e Cola)", command=self.copy).pack(side="left", padx=6)
        ttk.Button(btns, text="Fechar", command=self._on_close).pack(side="left", padx=6)

        self._center(parent)

    def copy(self):
        try:
            pyperclip.copy(self.qr_text)
            messagebox.showinfo("Copiado", "Código PIX copiado para a área de transferência.")
        except Exception as e:
            messagebox.showerror("Erro", f"Falha ao copiar: {e}")

    def _on_close(self):
        self.grab_release()
        self.destroy()

    def _center(self, parent):
        self.update_idletasks()
        if parent:
            px = parent.winfo_rootx()
            py = parent.winfo_rooty()
            pw = parent.winfo_width()
            ph = parent.winfo_height()
            w = self.winfo_width()
            h = self.winfo_height()
            x = px + (pw - w)//2
            y = py + (ph - h)//2
        else:
            sw = self.winfo_screenwidth()
            sh = self.winfo_screenheight()
            w = self.winfo_width()
            h = self.winfo_height()
            x = (sw - w)//2
            y = (sh - h)//2
        self.geometry(f"+{x}+{y}")
