# relatorios.py — Relatórios por Produto (um gráfico por vez + PDF/XLSX + Zoom efetivo)
# API esperada pelo PDV: open_relatorios(app, get_conn)

from __future__ import annotations
import os, sqlite3, traceback, math
from dataclasses import dataclass
from datetime import date, datetime, timedelta
from pathlib import Path
from typing import Callable, Optional, List, Tuple

import tkinter as tk
from tkinter import ttk, messagebox, filedialog

import pandas as pd
pd.options.display.float_format = "{:,.2f}".format

# Matplotlib -> Tkinter
import matplotlib
matplotlib.use("TkAgg")
import matplotlib.pyplot as plt
from matplotlib.backends.backend_tkagg import FigureCanvasTkAgg
from matplotlib.backends.backend_pdf import PdfPages

# XLSX opcional
try:
    import openpyxl  # noqa
    _XLSX_OK = True
except Exception:
    _XLSX_OK = False

OUTPUT_DIR = Path("./output")
CHART_DIR  = OUTPUT_DIR / "graficos"


def _ensure_dirs():
    OUTPUT_DIR.mkdir(parents=True, exist_ok=True)
    CHART_DIR.mkdir(parents=True, exist_ok=True)


def _today() -> str:
    return date.today().isoformat()


def _as_date(s: str, default: str) -> str:
    try:
        return datetime.fromisoformat((s or "").strip()).date().isoformat()
    except Exception:
        return default


def _brl(n: float) -> str:
    s = f"R$ {float(n or 0.0):,.2f}"
    return s.replace(",", "X").replace(".", ",").replace("X", ".")


# =============== Helpers de schema ===============

def _has(cx: sqlite3.Connection, name: str) -> bool:
    cur = cx.execute(
        "SELECT 1 FROM sqlite_master WHERE (type='table' OR type='view') AND lower(name)=lower(?)",
        (name,),
    )
    return cur.fetchone() is not None


def _has_col(cx: sqlite3.Connection, table: str, col: str) -> bool:
    try:
        cur = cx.execute(f'PRAGMA table_info("{table}")')
        return any((str(r[1]).lower() == col.lower()) for r in cur.fetchall())
    except Exception:
        return False


def _map_payment_label(s: pd.Series) -> pd.Series:
    return (
        s.astype(str)
        .str.lower()
        .map({"cash": "DINHEIRO", "card": "CARTÃO", "pix": "PIX"})
        .fillna("OUTROS")
    )


# =============== Coleta & Normalização ===============

@dataclass
class Frames:
    items: pd.DataFrame  # data_venda, produto, quantidade, valor_total, tipo_pagamento


def load_frames(cx: sqlite3.Connection, d1: Optional[str], d2: Optional[str]) -> Frames:
    parts: List[pd.DataFrame] = []

    # Modelo novo
    if _has(cx, "sale_items") and _has(cx, "products") and _has(cx, "sales"):
        sales_has_status = _has_col(cx, "sales", "status")
        sales_has_pm = _has_col(cx, "sales", "payment_method")
        status_expr = "UPPER(COALESCE(s.status,'PAGA'))" if sales_has_status else "'PAGA'"
        pm_expr = "COALESCE(s.payment_method,'cash')" if sales_has_pm else "'cash'"
        q = f"""
            SELECT
                DATE(s.ts)                   AS data_venda,
                COALESCE(p.name, p.sku)      AS produto,
                si.qty                        AS quantidade,
                (si.qty * si.unit_price)      AS valor_total,
                {pm_expr}                     AS payment_method,
                {status_expr}                 AS status_venda
            FROM sale_items si
            JOIN sales s    ON s.id = si.sale_id
            JOIN products p ON p.id = si.product_id
            WHERE s.ts IS NOT NULL
        """
        df = pd.read_sql(q, cx)
        df = df[df["status_venda"].astype(str).str.upper() == "PAGA"].copy()
        df["tipo_pagamento"] = _map_payment_label(df["payment_method"])  # normaliza
        df.drop(columns=["payment_method", "status_venda"], inplace=True, errors="ignore")
        parts.append(df)

    # Modelo antigo
    if _has(cx, "itens_venda") and _has(cx, "produtos") and _has(cx, "vendas"):
        v_has_status = _has_col(cx, "vendas", "status")
        v_has_pm = _has_col(cx, "vendas", "payment_method")
        v_has_pix = _has_col(cx, "vendas", "pagamento_pix")
        v_has_cartao = _has_col(cx, "vendas", "pagamento_cartao")
        v_has_dinheiro = _has_col(cx, "vendas", "pagamento_dinheiro")

        status_expr = "UPPER(COALESCE(v.status,'PAGA'))" if v_has_status else "'PAGA'"
        pm_expr = "COALESCE(v.payment_method,'')" if v_has_pm else "''"
        pix_expr = "COALESCE(v.pagamento_pix,0)" if v_has_pix else "0"
        car_expr = "COALESCE(v.pagamento_cartao,0)" if v_has_cartao else "0"
        din_expr = "COALESCE(v.pagamento_dinheiro,0)" if v_has_dinheiro else "0"

        q = f"""
            SELECT
                v.id                            AS venda_id,
                DATE(v.created_at)              AS data_venda,
                COALESCE(pr.nome, pr.sku)       AS produto,
                iv.quantidade                   AS quantidade,
                (iv.quantidade * iv.preco_unit) AS valor_total,
                {pm_expr}                       AS payment_method,
                {pix_expr}                      AS p_pix,
                {car_expr}                      AS p_cartao,
                {din_expr}                      AS p_dinheiro,
                {status_expr}                   AS status_venda
            FROM itens_venda iv
            JOIN vendas v   ON v.id = iv.venda_id
            JOIN produtos pr ON pr.id = iv.produto_id
            WHERE v.created_at IS NOT NULL
        """
        iv = pd.read_sql(q, cx)
        iv = iv[iv["status_venda"].astype(str).str.upper() == "PAGA"].copy()

        pm = iv["payment_method"].astype(str).str.upper().str.strip()
        tipo = pd.Series("OUTROS", index=iv.index)
        mask_pm = pm.isin(["PIX", "CARTÃO", "CARTAO", "DINHEIRO"])
        tipo.loc[mask_pm] = pm.loc[mask_pm].replace({"CARTAO": "CARTÃO"})

        vals = iv[["p_pix", "p_cartao", "p_dinheiro"]].astype(float)
        idx_max = vals.values.argmax(axis=1)
        labels = pd.Series(["PIX", "CARTÃO", "DINHEIRO"])
        mask_sem_pm = ~mask_pm & (vals.max(axis=1) > 0)
        if mask_sem_pm.any():
            tipo.loc[mask_sem_pm] = labels.iloc[idx_max[mask_sem_pm]].values

        iv["tipo_pagamento"] = tipo
        iv = iv.drop(
            columns=[
                "payment_method",
                "p_pix",
                "p_cartao",
                "p_dinheiro",
                "venda_id",
                "status_venda",
            ],
            errors="ignore",
        )
        parts.append(iv)

    parts = [p for p in parts if not p.empty]
    items = (
        pd.concat(parts, ignore_index=True)
        if parts
        else pd.DataFrame(columns=["data_venda", "produto", "quantidade", "valor_total", "tipo_pagamento"])
    )

    if not items.empty:
        if d1:
            items = items[items["data_venda"] >= d1]
        if d2:
            items = items[items["data_venda"] <= d2]
        items["quantidade"] = pd.to_numeric(items["quantidade"], errors="coerce").fillna(0.0)
        items["valor_total"] = pd.to_numeric(items["valor_total"], errors="coerce").fillna(0.0)
        items["produto"] = items["produto"].astype(str)
        items["tipo_pagamento"] = items["tipo_pagamento"].astype(str)

    return Frames(items=items)


# =============== Agregações ===============

@dataclass
class DashData:
    dinheiro: pd.DataFrame
    cartao: pd.DataFrame
    pix: pd.DataFrame
    total: pd.DataFrame
    total_valor: float
    total_qtd: float
    ticket: float
    top_produto: str


def build_dash(fr: Frames) -> DashData:
    it = fr.items.copy()
    if it.empty:
        vazio = pd.DataFrame(columns=["produto", "Valor Total", "Quantidade"])
        return DashData(vazio, vazio, vazio, vazio, 0.0, 0.0, 0.0, "-")

    def by_prod(df: pd.DataFrame) -> pd.DataFrame:
        g = (
            df.groupby("produto", as_index=False)
            .agg(**{"Valor Total": ("valor_total", "sum"), "Quantidade": ("quantidade", "sum")})
            .sort_values("Valor Total", ascending=False)
        )
        return g

    dinheiro = by_prod(it[it["tipo_pagamento"] == "DINHEIRO"])
    cartao = by_prod(it[it["tipo_pagamento"] == "CARTÃO"])
    pix = by_prod(it[it["tipo_pagamento"] == "PIX"])
    total = by_prod(it)

    total_valor = float(total["Valor Total"].sum()) if not total.empty else 0.0
    total_qtd = float(total["Quantidade"].sum()) if not total.empty else 0.0
    ticket = (total_valor / total_qtd) if total_qtd > 0 else 0.0
    top_produto = total.head(1)["produto"].iloc[0] if not total.empty else "-"

    return DashData(dinheiro, cartao, pix, total, total_valor, total_qtd, ticket, top_produto)


# =============== Painel Único (um gráfico por vez) ===============

CHART_TYPES = ["Barras", "Barras horizontais", "Linha", "Pizza"]
SALE_TYPES = ["DINHEIRO", "CARTÃO", "PIX", "TOTAL"]


class SingleChartPanel:
    def __init__(self, parent: tk.Widget):
        self.frame = ttk.Frame(parent, padding=6)

        # Barra de controles do painel
        ctrl = ttk.Frame(self.frame)
        ctrl.pack(fill="x")
        ttk.Label(ctrl, text="Tipo de gráfico:").pack(side="left")
        self.cmb_kind = ttk.Combobox(ctrl, values=CHART_TYPES, state="readonly", width=22)
        self.cmb_kind.current(0)
        self.cmb_kind.pack(side="left", padx=6)

        ttk.Label(ctrl, text="Venda:").pack(side="left", padx=(10, 0))
        self.cmb_sale = ttk.Combobox(ctrl, values=SALE_TYPES, state="readonly", width=12)
        self.cmb_sale.current(3)  # TOTAL
        self.cmb_sale.pack(side="left", padx=6)

        ttk.Label(ctrl, text="Top N:").pack(side="left", padx=(10, 0))
        self.spn_top = ttk.Spinbox(ctrl, from_=5, to=50, width=5)
        self.spn_top.delete(0, "end")
        self.spn_top.insert(0, "20")
        self.spn_top.pack(side="left", padx=6)

        self.btn_save = ttk.Button(ctrl, text="Salvar PNG", command=self.save_png)
        self.btn_save.pack(side="right")

        # Zoom (slider + botões +/−)
        zoom_box = ttk.Frame(self.frame)
        zoom_box.pack(fill="x", pady=(4, 0))
        ttk.Label(zoom_box, text="Zoom:").pack(side="left")
        self.zoom_var = tk.DoubleVar(value=1.0)
        self.zoom_lbl = ttk.Label(zoom_box, width=6, anchor="w", text="100%")
        self.zoom = ttk.Scale(
            zoom_box,
            from_=0.5,
            to=3.0,
            orient="horizontal",
            variable=self.zoom_var,
            command=self._on_zoom_change,
        )
        self.zoom.pack(side="left", fill="x", expand=True, padx=6)
        self.zoom_lbl.pack(side="left")

        # botões +/−
        self.btn_minus = ttk.Button(zoom_box, text="−", width=3, command=lambda: self._bump_zoom(-0.1))
        self.btn_plus = ttk.Button(zoom_box, text="+", width=3, command=lambda: self._bump_zoom(+0.1))
        self.btn_minus.pack(side="left", padx=(6, 2))
        self.btn_plus.pack(side="left")

        # Canvas
        self.canvas_holder = ttk.Frame(self.frame)
        self.canvas_holder.pack(fill="both", expand=True, pady=(6, 0))

        # Estados
        self._fig: Optional[plt.Figure] = None
        self._canvas: Optional[FigureCanvasTkAgg] = None
        self._df_map: dict[str, pd.DataFrame] = {k: pd.DataFrame() for k in SALE_TYPES}
        self._title_map: dict[str, str] = {
            "DINHEIRO": "Vendas DINHEIRO — por produto",
            "CARTÃO": "Vendas CARTÃO — por produto",
            "PIX": "Vendas PIX — por produto",
            "TOTAL": "TOTAL (DINHEIRO+CARTÃO+PIX) — por produto",
        }

        # Bindings
        self.cmb_kind.bind("<<ComboboxSelected>>", lambda e: self.render())
        self.cmb_sale.bind("<<ComboboxSelected>>", lambda e: self.render())
        self.spn_top.bind("<Return>", lambda e: self.render())
        self.spn_top.bind("<FocusOut>", lambda e: self.render())

        # atalhos de teclado para zoom
        self.frame.bind_all("<Control-minus>", lambda e: self._bump_zoom(-0.1))
        self.frame.bind_all("<Control-KP_Subtract>", lambda e: self._bump_zoom(-0.1))
        self.frame.bind_all("<Control-equal>", lambda e: self._bump_zoom(+0.1))
        self.frame.bind_all("<Control-KP_Add>", lambda e: self._bump_zoom(+0.1))

    # ---- callbacks
    def _bump_zoom(self, delta: float):
        z = float(self.zoom_var.get() or 1.0)
        z = max(0.5, min(3.0, z + delta))
        self.zoom_var.set(z)
        self._on_zoom_change()

    def _on_zoom_change(self, *_):
        z = float(self.zoom_var.get() or 1.0)
        self.zoom_lbl.config(text=f"{int(round(z * 100))}%")
        self.render()

    def set_data(self, dinheiro: pd.DataFrame, cartao: pd.DataFrame, pix: pd.DataFrame, total: pd.DataFrame):
        self._df_map["DINHEIRO"] = dinheiro
        self._df_map["CARTÃO"] = cartao
        self._df_map["PIX"] = pix
        self._df_map["TOTAL"] = total
        self.render()

    def _current_df_title(self) -> Tuple[pd.DataFrame, str]:
        sale = self.cmb_sale.get() or "TOTAL"
        df = self._df_map.get(sale, pd.DataFrame()).copy()
        title = self._title_map.get(sale, "Vendas — por produto")
        return df, title

    def render(self):
        df, title = self._current_df_title()
        try:
            topn = max(1, int(self.spn_top.get()))
        except Exception:
            topn = 20
        df = df.head(topn)

        # limpar anterior
        if self._canvas:
            try:
                self._canvas.get_tk_widget().destroy()
            except Exception:
                pass
            self._canvas = None
        if self._fig is not None:
            plt.close(self._fig)
            self._fig = None

        kind = self.cmb_kind.get() or "Barras"
        z = float(self.zoom_var.get() or 1.0)

        # tamanhos base menores para caber no viewport e permitir textos
        base_w, base_h = 6.0, 3.4
        if kind == "Barras horizontais":
            base_h = max(3.4, 0.30 * len(df))
        fig_w, fig_h = base_w * z, base_h * z
        fs_title = 12 * z
        fs_label = 10 * z
        fs_ticks = 9 * z

        if kind == "Barras":
            fig, ax = plt.subplots(figsize=(fig_w, fig_h))
            ax.bar(df["produto"], df["Valor Total"])
            ax.set_xlabel("Produto", fontsize=fs_label)
            ax.set_ylabel("Total (R$)", fontsize=fs_label)
            ax.set_title(title, fontsize=fs_title)
            ax.tick_params(axis="x", rotation=45, labelsize=fs_ticks)
            ax.tick_params(axis="y", labelsize=fs_ticks)

        elif kind == "Barras horizontais":
            fig, ax = plt.subplots(figsize=(fig_w, fig_h))
            d = df.iloc[::-1]
            ax.barh(d["produto"], d["Valor Total"])
            ax.set_xlabel("Total (R$)", fontsize=fs_label)
            ax.set_ylabel("Produto", fontsize=fs_label)
            ax.set_title(title, fontsize=fs_title)
            ax.tick_params(axis="x", labelsize=fs_ticks)
            ax.tick_params(axis="y", labelsize=fs_ticks)

        elif kind == "Linha":
            fig, ax = plt.subplots(figsize=(fig_w, fig_h))
            ax.plot(range(len(df)), df["Valor Total"], marker="o")
            ax.set_xticks(range(len(df)))
            ax.set_xticklabels(df["produto"], rotation=45, ha="right", fontsize=fs_ticks)
            ax.set_xlabel("Produto", fontsize=fs_label)
            ax.set_ylabel("Total (R$)", fontsize=fs_label)
            ax.set_title(title, fontsize=fs_title)
            ax.tick_params(axis="y", labelsize=fs_ticks)

        else:  # Pizza
            fig, ax = plt.subplots(figsize=(6 * z, 6 * z))
            ax.pie(
                df["Valor Total"],
                labels=df["produto"],
                autopct="%1.1f%%",
                startangle=90,
                textprops={"fontsize": fs_ticks},
            )
            ax.axis("equal")
            ax.set_title(title, fontsize=fs_title)

        # melhora o encaixe para aparecer todo o texto abaixo
        fig.tight_layout()

        self._fig = fig
        self._canvas = FigureCanvasTkAgg(fig, master=self.canvas_holder)
        self._canvas.draw()
        self._canvas.get_tk_widget().pack(fill="both", expand=True)

    def save_png(self):
        if self._fig is None:
            messagebox.showinfo("Gráfico", "Nada para salvar.")
            return
        _ensure_dirs()
        sale = (self.cmb_sale.get() or "TOTAL").lower()
        kind = (self.cmb_kind.get() or "barras").lower().replace(" ", "_")
        path = CHART_DIR / f"grafico_{sale}_{kind}.png"
        self._fig.tight_layout()
        self._fig.savefig(path, dpi=300, bbox_inches="tight")
        messagebox.showinfo("Gráfico", f"Salvo em: {path}")


# =============== Exportadores (PDF / XLSX) ===============

def _kpi_summary_text(total_valor: float, total_qtd: float, ticket: float, top_produto: str) -> str:
    return (
        f"Total: {_brl(total_valor)}    "
        f"Volume: {total_qtd:,.0f}".replace(",", ".")
        + f"    Ticket médio: {_brl(ticket)}    "
        f"Top produto: {top_produto}"
    )


def _draw_table_page(ax, df: pd.DataFrame, title: str):
    ax.axis("off")
    ax.set_title(title, fontsize=12, pad=10, loc="left")
    table = ax.table(cellText=df.values, colLabels=list(df.columns), cellLoc="right", loc="upper left")
    table.auto_set_font_size(False)
    table.set_fontsize(8)
    table.scale(1.0, 1.2)


def _export_pdf(path: str, frames: Frames, dash: DashData, fig_to_include: Optional[plt.Figure]):
    items = frames.items.copy()
    items = (
        items[["data_venda", "produto", "quantidade", "valor_total", "tipo_pagamento"]]
        .rename(
            columns={
                "data_venda": "Data",
                "produto": "Produto",
                "quantidade": "Qtd",
                "valor_total": "Total (R$)",
                "tipo_pagamento": "Pagamento",
            }
        )
        .sort_values(["Data", "Produto"])
        .reset_index(drop=True)
    )
    items["Total (R$)"] = items["Total (R$)"].map(lambda v: float(f"{float(v):.2f}"))
    items["Qtd"] = items["Qtd"].map(lambda v: float(f"{float(v):.2f}"))

    with PdfPages(path) as pdf:
        # KPIs
        fig, ax = plt.subplots(figsize=(8.27, 11.69))
        ax.axis("off")
        ax.text(0.03, 0.95, "Relatório de Vendas — Detalhado", fontsize=16, weight="bold", ha="left", va="top")
        ax.text(0.03, 0.90, _kpi_summary_text(dash.total_valor, dash.total_qtd, dash.ticket, dash.top_produto), fontsize=11, ha="left", va="top")
        ax.text(0.03, 0.86, f"Gerado em: {datetime.now().strftime('%d/%m/%Y %H:%M:%S')}", fontsize=9)
        pdf.savefig(fig, bbox_inches="tight")
        plt.close(fig)

        # Tabelas (~40 linhas/página)
        rows_per_page = 40
        total_pages = max(1, math.ceil(len(items) / rows_per_page))
        for i in range(total_pages):
            chunk = items.iloc[i * rows_per_page : (i + 1) * rows_per_page]
            fig, ax = plt.subplots(figsize=(8.27, 11.69))
            _draw_table_page(ax, chunk, f"Vendas (página {i + 1}/{total_pages})")
            pdf.savefig(fig, bbox_inches="tight")
            plt.close(fig)

        # Apenas o gráfico atual (se existir)
        if fig_to_include is not None:
            fig_to_include.tight_layout()
            pdf.savefig(fig_to_include, bbox_inches="tight")


def _export_xlsx(path: str, frames: Frames, dash: DashData):
    items = frames.items.copy()
    det = (
        items[["data_venda", "produto", "quantidade", "valor_total", "tipo_pagamento"]]
        .rename(
            columns={
                "data_venda": "Data",
                "produto": "Produto",
                "quantidade": "Qtd",
                "valor_total": "Total (R$)",
                "tipo_pagamento": "Pagamento",
            }
        )
        .sort_values(["Data", "Produto"])
    )

    resumo = (
        items.groupby("tipo_pagamento", as_index=False)
        .agg(Quantidade=("quantidade", "sum"), Valor_Total=("valor_total", "sum"))
        .rename(columns={"tipo_pagamento": "Pagamento"})
        .sort_values("Valor_Total", ascending=False)
    )
    resumo_total = pd.DataFrame(
        [
            {
                "Pagamento": "TOTAL",
                "Quantidade": float(resumo["Quantidade"].sum() if not resumo.empty else 0.0),
                "Valor_Total": float(resumo["Valor_Total"].sum() if not resumo.empty else 0.0),
            }
        ]
    )
    resumo = pd.concat([resumo, resumo_total], ignore_index=True)

    with pd.ExcelWriter(path, engine="openpyxl") as writer:
        det.to_excel(writer, index=False, sheet_name="Detalhe")
        resumo.to_excel(writer, index=False, sheet_name="Resumo")


# =============== Janela principal ===============

class ReportsWindow:
    def __init__(self, app, get_conn: Callable[[], sqlite3.Connection]):
        self.app = app
        self.get_conn = get_conn
        self.root = tk.Toplevel(app.root)
        self.root.title("Relatórios de Vendas — Por Produto (um gráfico por vez)")
        self.root.geometry("1280x850")  # fallback inicial
        self.root.resizable(True, True)   # permitir maximizar/minimizar
        try:
            self.root.transient(app.root)
        except Exception:
            pass

        # abre maximizada (Windows)
        try:
            self.root.after(50, lambda: self.root.state('zoomed'))
        except Exception:
            # fallback para telas que não suportam 'zoomed'
            sw = self.root.winfo_screenwidth()
            sh = self.root.winfo_screenheight()
            self.root.geometry(f"{sw}x{sh}+0+0")

        # Título
        top = ttk.Frame(self.root, padding=10)
        top.pack(fill="x")
        ttk.Label(top, text="Relatórios de Vendas por Produto", font=("Segoe UI", 16, "bold")).pack(side="left")

        # Barra superior (período + ações)
        bar = ttk.Frame(self.root, padding=(10, 0))
        bar.pack(fill="x")
        ttk.Label(bar, text="Início:").pack(side="left")
        self.ent_ini = ttk.Entry(bar, width=12)
        self.ent_ini.pack(side="left", padx=(4, 8))
        ttk.Label(bar, text="Fim:").pack(side="left")
        self.ent_fim = ttk.Entry(bar, width=12)
        self.ent_fim.pack(side="left", padx=(4, 8))
        self.ent_ini.insert(0, (date.today() - timedelta(days=30)).isoformat())
        self.ent_fim.insert(0, _today())

        ttk.Button(bar, text="Gerar", command=self._run).pack(side="left", padx=8)
        ttk.Button(bar, text="Salvar PDF…", command=self._on_save_pdf).pack(side="left", padx=(12, 4))
        ttk.Button(bar, text="Salvar Planilha…", command=self._on_save_sheet).pack(side="left")

        self.var_status = tk.StringVar(value="Pronto.")
        ttk.Label(bar, textvariable=self.var_status).pack(side="right")

        # KPIs
        kpi = ttk.Frame(self.root, padding=(10, 6))
        kpi.pack(fill="x")
        self.lbl_total = ttk.Label(kpi, text="Total: R$ 0,00", font=("Segoe UI", 11, "bold"))
        self.lbl_volume = ttk.Label(kpi, text="Volume: 0", font=("Segoe UI", 11, "bold"))
        self.lbl_ticket = ttk.Label(kpi, text="Ticket médio: R$ 0,00", font=("Segoe UI", 11, "bold"))
        self.lbl_topprd = ttk.Label(kpi, text="Top produto: -", font=("Segoe UI", 11))
        for w in (self.lbl_total, self.lbl_volume, self.lbl_ticket, self.lbl_topprd):
            w.pack(side="left", padx=12)

        # Painel Único
        self.panel = SingleChartPanel(self.root)
        self.panel.frame.pack(fill="both", expand=True, padx=10, pady=8)

        # Estados
        self._frames: Optional[Frames] = None
        self._dash: Optional[DashData] = None

    def _run(self):
        _ensure_dirs()
        d1 = _as_date(self.ent_ini.get(), (date.today() - timedelta(days=30)).isoformat())
        d2 = _as_date(self.ent_fim.get(), _today())
        try:
            with self.get_conn() as cx:
                fr = load_frames(cx, d1, d2)
            dash = build_dash(fr)
            self._frames, self._dash = fr, dash

            # KPIs
            self.lbl_total.config(text=f"Total: {_brl(dash.total_valor)}")
            self.lbl_volume.config(text=f"Volume: {dash.total_qtd:,.0f}".replace(",", "."))
            self.lbl_ticket.config(text=f"Ticket médio: {_brl(dash.ticket)}")
            self.lbl_topprd.config(text=f"Top produto: {dash.top_produto}")

            # Alimenta o painel único
            self.panel.set_data(dash.dinheiro, dash.cartao, dash.pix, dash.total)

            self.var_status.set("Relatório gerado.")
        except Exception as e:
            self.var_status.set("Falha ao gerar.")
            messagebox.showerror("Relatórios", f"Erro: {e}\n\n{traceback.format_exc()}")

    # ======= Exportações =======
    def _default_filename_base(self) -> str:
        d1 = _as_date(self.ent_ini.get(), (date.today() - timedelta(days=30)).isoformat())
        d2 = _as_date(self.ent_fim.get(), _today())
        return f"vendas_{d1}_a_{d2}"

    def _on_save_pdf(self):
        if not self._frames or self._frames.items.empty or not self._dash:
            messagebox.showinfo("PDF", "Gere o relatório primeiro.")
            return
        base = self._default_filename_base()
        path = filedialog.asksaveasfilename(
            title="Salvar PDF",
            defaultextension=".pdf",
            initialfile=f"{base}.pdf",
            filetypes=[("PDF", "*.pdf")],
        )
        if not path:
            return
        try:
            fig = self.panel._fig
            _export_pdf(path, self._frames, self._dash, fig)
            messagebox.showinfo("PDF", f"PDF salvo em:\n{path}")
        except Exception as e:
            messagebox.showerror("PDF", f"Falha ao gerar PDF:\n{e}\n\n{traceback.format_exc()}")

    def _on_save_sheet(self):
        if not self._frames or self._frames.items.empty or not self._dash:
            messagebox.showinfo("Planilha", "Gere o relatório primeiro.")
            return
        base = self._default_filename_base()
        if _XLSX_OK:
            path = filedialog.asksaveasfilename(
                title="Salvar Planilha XLSX",
                defaultextension=".xlsx",
                initialfile=f"{base}.xlsx",
                filetypes=[("Planilha Excel", "*.xlsx")],
            )
            if not path:
                return
            try:
                _export_xlsx(path, self._frames, self._dash)
                messagebox.showinfo("Planilha", f"Planilha salva em:\n{path}")
            except Exception as e:
                messagebox.showerror("Planilha", f"Falha ao gerar XLSX:\n{e}\n\n{traceback.format_exc()}")
        else:
            path = filedialog.asksaveasfilename(
                title="Salvar CSV (openpyxl ausente)",
                defaultextension=".csv",
                initialfile=f"{base}.csv",
                filetypes=[("CSV", "*.csv")],
            )
            if not path:
                return
            try:
                items = self._frames.items.copy()
                items.to_csv(path, sep=";", index=False, encoding="utf-8-sig")
                messagebox.showinfo(
                    "Planilha",
                    f"CSV salvo em:\n{path}\n\n"
                    "Dica: instale 'openpyxl' para exportar XLSX:\n"
                    "pip install openpyxl",
                )
            except Exception as e:
                messagebox.showerror("Planilha", f"Falha ao gerar CSV:\n{e}\n\n{traceback.format_exc()}")


# ======== API p/ a tela principal ========

def open_relatorios(app, get_conn: Callable[[], sqlite3.Connection]):
    return ReportsWindow(app, get_conn)


def open_reports(app, get_conn: Callable[[], sqlite3.Connection]):
    return open_relatorios(app, get_conn)
