import os
import sys
import argparse
import random
import string
import time
from datetime import datetime, timezone, timedelta

# Importa do app para reutilizar conexão / init / funções
try:
    from erp_pdv_tk import (
        get_conn,
        init_db,
        create_user,
        set_user_perms,
        AVAILABLE_PERMS,
        get_db_path,
    )
except Exception as e:
    print("Erro: não consegui importar 'erp_pdv_tk'. Garanta que seed_db.py está na mesma pasta do erp_pdv_tk.py.")
    raise

# ------------------ Helpers de geração ------------------

FIRST_NAMES = [
    "Ana","Bruno","Carlos","Daniela","Eduardo","Fernanda","Gabriel","Helena","Igor","Joana","Kleber",
    "Larissa","Marcelo","Natália","Otávio","Paula","Rafael","Sabrina","Thiago","Viviane","William","Yasmin"
]
LAST_NAMES = [
    "Almeida","Barbosa","Cardoso","Dias","Esteves","Ferreira","Gomes","Henrique","Iraê","Jesus","Lima",
    "Mendes","Nogueira","Oliveira","Pereira","Queiroz","Ramos","Silva","Teixeira","Uchoa","Vieira","Xavier","Zago"
]

def rand_name():
    return f"{random.choice(FIRST_NAMES)} {random.choice(LAST_NAMES)}"

def rand_email(name, idx):
    base = name.lower().replace(" ", ".").replace("ç","c").replace("ã","a").replace("á","a").replace("é","e")
    return f"{base}.{idx}@example.com"

def rand_phone():
    return f"+55{random.randint(11,99)}9{random.randint(1000,9999)}{random.randint(1000,9999)}"

def rand_sku(i):
    return f"SKU-{i:07d}"

def rand_price():
    return round(random.uniform(2.0, 500.0), 2)

def chunks(iterable, size):
    buf = []
    for x in iterable:
        buf.append(x)
        if len(buf) >= size:
            yield buf
            buf = []
    if buf:
        yield buf

# ------------------ Schema extra opcional ------------------

EXTRA_SCHEMA_SQL = """
CREATE TABLE IF NOT EXISTS products (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    sku TEXT UNIQUE NOT NULL,
    name TEXT NOT NULL,
    unit_price REAL NOT NULL,
    stock_qty INTEGER NOT NULL DEFAULT 0
);
CREATE INDEX IF NOT EXISTS ix_products_sku ON products(sku);

CREATE TABLE IF NOT EXISTS customers (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    name TEXT NOT NULL,
    email TEXT,
    phone TEXT
);

CREATE TABLE IF NOT EXISTS sales (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    customer_id INTEGER,
    ts TEXT NOT NULL,
    total_gross REAL NOT NULL,
    FOREIGN KEY(customer_id) REFERENCES customers(id)
);

CREATE TABLE IF NOT EXISTS sale_items (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    sale_id INTEGER NOT NULL,
    product_id INTEGER NOT NULL,
    qty INTEGER NOT NULL,
    unit_price REAL NOT NULL,
    FOREIGN KEY(sale_id) REFERENCES sales(id) ON DELETE CASCADE,
    FOREIGN KEY(product_id) REFERENCES products(id)
);
CREATE INDEX IF NOT EXISTS ix_sale_items_sale ON sale_items(sale_id);
"""

def apply_fast_pragmas(conn):
    # Para geração massiva de dados (rápida, menos durável). Use só em testes.
    conn.execute("PRAGMA journal_mode = WAL;")
    conn.execute("PRAGMA synchronous = OFF;")
    conn.execute("PRAGMA temp_store = MEMORY;")
    conn.execute("PRAGMA mmap_size = 30000000000;")  # tenta mmmap (safe se não suportar)

# ------------------ Seeders ------------------

def seed_users(total_users: int, admin_count: int, start_index: int = 1):
    """Cria usuários em massa (admin + comuns) e define permissões aleatórias para não-admin."""
    assert admin_count <= total_users, "admins não pode ser maior que total de usuários"
    print(f"[users] criando {total_users} usuários ({admin_count} admins)...")
    created = 0
    admin_left = admin_count

    # Criar usuários sequenciais: user00001 ...
    for i in range(start_index, start_index + total_users):
        username = f"user{i:05d}"
        full = rand_name()
        is_admin = admin_left > 0
        try:
            uid = create_user(username, full, "1234", is_admin=is_admin, active=True)
            created += 1
            if is_admin:
                admin_left -= 1
            else:
                # Permissões aleatórias (~50% de chance por perm)
                mapping = {k: (random.random() < 0.5) for k in AVAILABLE_PERMS.keys()}
                set_user_perms(uid, mapping)
        except Exception as e:
            # Pode ocorrer UNIQUE(username) se rodar mais de uma vez com o mesmo prefixo
            print(f"  - aviso: não criei '{username}': {e}")

    print(f"[users] prontos: {created} usuários criados.")


def ensure_extra_schema():
    with get_conn() as conn:
        conn.executescript(EXTRA_SCHEMA_SQL)


def seed_products(n_products: int, batch_size: int = 5000):
    print(f"[products] criando {n_products} produtos...")
    with get_conn() as conn:
        for batch in chunks(range(1, n_products + 1), batch_size):
            rows = [(rand_sku(i), f"Produto {i:05d}", rand_price(), random.randint(0, 1000)) for i in batch]
            conn.executemany(
                "INSERT OR IGNORE INTO products (sku, name, unit_price, stock_qty) VALUES (?,?,?,?)",
                rows,
            )
    print("[products] ok")


def seed_customers(n_customers: int, batch_size: int = 5000):
    print(f"[customers] criando {n_customers} clientes...")
    with get_conn() as conn:
        for batch in chunks(range(1, n_customers + 1), batch_size):
            names = [rand_name() for _ in batch]
            rows = [(nm, rand_email(nm, i), rand_phone()) for nm, i in zip(names, batch)]
            conn.executemany(
                "INSERT INTO customers (name, email, phone) VALUES (?,?,?)",
                rows,
            )
    print("[customers] ok")


def _list_ids(conn, table, col="id"):
    cur = conn.execute(f"SELECT {col} FROM {table}")
    return [r[0] for r in cur.fetchall()]

def seed_sales(n_sales: int, max_items: int = 5, back_days: int = 120, batch_size: int = 1000):
    print(f"[sales] criando {n_sales} vendas (até {max_items} itens cada)...")
    rng = random.Random(42)

    with get_conn() as conn:
        product_ids = _list_ids(conn, "products")
        customer_ids = _list_ids(conn, "customers")
        if not product_ids or not customer_ids:
            print("  - você precisa ter products e customers antes de gerar sales")
            return

    def rand_ts():
        delta = timedelta(days=rng.randint(0, back_days), seconds=rng.randint(0, 86400))
        return (datetime.now(timezone.utc) - delta).isoformat()

    sales_rows = []
    with get_conn() as conn:
        for b in chunks(range(1, n_sales + 1), batch_size):
            sales_rows.clear()
            for _ in b:
                cust = rng.choice(customer_ids)
                ts = rand_ts()
                # cria a venda com total 0 e atualiza após inserir itens
                conn.execute("INSERT INTO sales (customer_id, ts, total_gross) VALUES (?,?,?)", (cust, ts, 0.0))
                sale_id = conn.execute("SELECT last_insert_rowid()").fetchone()[0]
                n_items = rng.randint(1, max_items)
                total = 0.0
                for __ in range(n_items):
                    pid = rng.choice(product_ids)
                    qty = rng.randint(1, 5)
                    # pega preço atual do produto
                    cur = conn.execute("SELECT unit_price FROM products WHERE id=?", (pid,))
                    unit_price = float(cur.fetchone()[0])
                    total += unit_price * qty
                    conn.execute(
                        "INSERT INTO sale_items (sale_id, product_id, qty, unit_price) VALUES (?,?,?,?)",
                        (sale_id, pid, qty, unit_price),
                    )
                conn.execute("UPDATE sales SET total_gross=? WHERE id=?", (round(total, 2), sale_id))
    print("[sales] ok")

# ------------------ CLI ------------------

def main():
    parser = argparse.ArgumentParser(description="Seed de dados massivos para ERP/PDV (SQLite)")
    parser.add_argument("--db", help="Caminho do .db (opcional; padrão: ERP_PDV_DB ou erp_pdv.db)")
    parser.add_argument("--users", type=int, default=1000, help="Quantidade de usuários (default: 1000)")
    parser.add_argument("--admins", type=int, default=10, help="Quantidade de admins dentro dos usuários (default: 10)")
    parser.add_argument("--start-index", type=int, default=1, help="Índice inicial dos usuários sequenciais (default: 1)")

    parser.add_argument("--with-extras", action="store_true", help="Criar/popular tabelas extras (produtos, clientes, vendas)")
    parser.add_argument("--products", type=int, default=5000, help="Qtde de produtos quando --with-extras (default: 5000)")
    parser.add_argument("--customers", type=int, default=2000, help="Qtde de clientes quando --with-extras (default: 2000)")
    parser.add_argument("--sales", type=int, default=10000, help="Qtde de vendas quando --with-extras (default: 10000)")
    parser.add_argument("--max-items", type=int, default=5, help="Máx itens por venda quando --with-extras (default: 5)")

    parser.add_argument("--fast", action="store_true", help="PRAGMAs agressivos para popular mais rápido (uso local)")
    args = parser.parse_args()

    if args.db:
        os.environ["ERP_PDV_DB"] = args.db

    # Garante schema base e admin
    init_db()

    # PRAGMAs de performance (opcional)
    if args.fast:
        with get_conn() as conn:
            apply_fast_pragmas(conn)

    t0 = time.time()

    # Usuários + permissões
    seed_users(args.users, args.admins, start_index=args.start_index)

    if args.with_extras:
        ensure_extra_schema()
        if args.fast:
            with get_conn() as conn:
                apply_fast_pragmas(conn)
        seed_products(args.products)
        seed_customers(args.customers)
        seed_sales(args.sales, max_items=args.max_items)

    dt = time.time() - t0
    print(f"\n[done] carga finalizada em {dt:.1f}s")
    print(f"Banco em uso: {os.environ.get('ERP_PDV_DB') or get_db_path()}")

if __name__ == "__main__":
    main()
