import os
import argparse
import random
from datetime import datetime, timezone, timedelta

# Reutiliza as conexões e inicialização do seu app
from erp_pdv_tk import get_conn, init_db, get_db_path

# ------------------ schema extra (produtos/clientes/vendas) ------------------

EXTRA_SCHEMA_SQL = """
CREATE TABLE IF NOT EXISTS products (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    sku TEXT UNIQUE NOT NULL,
    name TEXT NOT NULL,
    unit_price REAL NOT NULL,
    stock_qty INTEGER NOT NULL DEFAULT 0
);
CREATE INDEX IF NOT EXISTS ix_products_sku ON products(sku);

CREATE TABLE IF NOT EXISTS customers (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    name TEXT NOT NULL,
    email TEXT,
    phone TEXT
);
-- índice opcional (não obrigatório)
CREATE INDEX IF NOT EXISTS ix_customers_email ON customers(email);

CREATE TABLE IF NOT EXISTS sales (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    customer_id INTEGER,
    ts TEXT NOT NULL,
    total_gross REAL NOT NULL,
    FOREIGN KEY(customer_id) REFERENCES customers(id)
);

CREATE TABLE IF NOT EXISTS sale_items (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    sale_id INTEGER NOT NULL,
    product_id INTEGER NOT NULL,
    qty INTEGER NOT NULL,
    unit_price REAL NOT NULL,
    FOREIGN KEY(sale_id) REFERENCES sales(id) ON DELETE CASCADE,
    FOREIGN KEY(product_id) REFERENCES products(id)
);
CREATE INDEX IF NOT EXISTS ix_sale_items_sale ON sale_items(sale_id);
"""

def apply_fast_pragmas(conn):
    # acelera bastante inserções (use só em testes)
    conn.execute("PRAGMA journal_mode = WAL;")
    conn.execute("PRAGMA synchronous = OFF;")
    conn.execute("PRAGMA temp_store = MEMORY;")

# ------------------ helpers ------------------

FIRST = ["Ana","Bruno","Carlos","Daniela","Eduardo","Fernanda","Gabriel","Helena","Igor","Joana","Larissa","Marcelo",
         "Natália","Otávio","Paula","Rafael","Sabrina","Thiago","Viviane","William","Yasmin"]
LAST = ["Almeida","Barbosa","Cardoso","Dias","Esteves","Ferreira","Gomes","Lima","Mendes","Nogueira","Oliveira",
        "Pereira","Queiroz","Ramos","Silva","Teixeira","Vieira","Xavier"]

def rand_name(rng):
    return f"{rng.choice(FIRST)} {rng.choice(LAST)}"

def rand_email(name, idx):
    base = (name.lower().replace(" ", ".")
            .replace("ç","c").replace("ã","a").replace("á","a")
            .replace("é","e").replace("ê","e").replace("í","i").replace("ó","o").replace("ú","u"))
    return f"{base}.{idx}@example.com"

def rand_phone(rng):
    return f"+55{rng.randint(11,99)}9{rng.randint(1000,9999)}{rng.randint(1000,9999)}"

def rand_sku(i): return f"SKU-{i:07d}"
def rand_price(rng): return round(rng.uniform(2.0, 500.0), 2)

def chunks(iterable, size):
    buf = []
    for x in iterable:
        buf.append(x)
        if len(buf) >= size:
            yield buf
            buf = []
    if buf:
        yield buf

# ------------------ seeders ------------------

def ensure_extra_schema():
    with get_conn() as conn:
        conn.executescript(EXTRA_SCHEMA_SQL)

def seed_products(n_products: int, batch_size: int = 5000, seed: int = 42):
    print(f"[products] criando {n_products} produtos...")
    rng = random.Random(seed)
    with get_conn() as conn:
        for batch in chunks(range(1, n_products + 1), batch_size):
            rows = [(rand_sku(i), f"Produto {i:05d}", rand_price(rng), rng.randint(0, 1000)) for i in batch]
            conn.executemany(
                "INSERT OR IGNORE INTO products (sku, name, unit_price, stock_qty) VALUES (?,?,?,?)",
                rows,
            )
    print("[products] ok")

def seed_customers(n_customers: int, batch_size: int = 5000, seed: int = 99):
    print(f"[customers] criando {n_customers} clientes...")
    rng = random.Random(seed)
    with get_conn() as conn:
        for batch in chunks(range(1, n_customers + 1), batch_size):
            names = [rand_name(rng) for _ in batch]
            rows = [(nm, rand_email(nm, i), rand_phone(rng)) for nm, i in zip(names, batch)]
            conn.executemany("INSERT INTO customers (name, email, phone) VALUES (?,?,?)", rows)
    print("[customers] ok")

def _list_ids(conn, table, col="id"):
    cur = conn.execute(f"SELECT {col} FROM {table}")
    return [r[0] for r in cur.fetchall()]

def seed_sales(n_sales: int, max_items: int = 5, back_days: int = 120, batch_size: int = 1000, seed: int = 1234):
    print(f"[sales] criando {n_sales} vendas (até {max_items} itens cada)...")
    rng = random.Random(seed)

    with get_conn() as conn:
        product_ids = _list_ids(conn, "products")
        customer_ids = _list_ids(conn, "customers")
        if not product_ids:
            print("  ! você precisa ter 'products' antes de gerar 'sales'")
            return
        if not customer_ids:
            print("  ! não há 'customers'; gerando 100 por padrão...")
            conn.execute("INSERT INTO customers (name) VALUES ('Cliente Padrão 1')")
            customer_ids = _list_ids(conn, "customers")

    def rand_ts():
        delta = timedelta(days=rng.randint(0, back_days), seconds=rng.randint(0, 86400))
        return (datetime.now(timezone.utc) - delta).isoformat()

    with get_conn() as conn:
        for b in chunks(range(1, n_sales + 1), batch_size):
            for _ in b:
                cust = rng.choice(customer_ids) if customer_ids else None
                ts = rand_ts()
                conn.execute("INSERT INTO sales (customer_id, ts, total_gross) VALUES (?,?,?)", (cust, ts, 0.0))
                sale_id = conn.execute("SELECT last_insert_rowid()").fetchone()[0]
                n_items = rng.randint(1, max_items)
                total = 0.0
                for __ in range(n_items):
                    pid = rng.choice(product_ids)
                    qty = rng.randint(1, 5)
                    unit_price = float(conn.execute("SELECT unit_price FROM products WHERE id=?", (pid,)).fetchone()[0])
                    total += unit_price * qty
                    conn.execute(
                        "INSERT INTO sale_items (sale_id, product_id, qty, unit_price) VALUES (?,?,?,?)",
                        (sale_id, pid, qty, unit_price),
                    )
                conn.execute("UPDATE sales SET total_gross=? WHERE id=?", (round(total, 2), sale_id))
    print("[sales] ok")

# ------------------ CLI ------------------

def main():
    ap = argparse.ArgumentParser(description="Popular SQLite com produtos + vendas (dados de teste)")
    ap.add_argument("--db", help="Caminho do .db (padrão: ERP_PDV_DB ou erp_pdv.db)")
    ap.add_argument("--products", type=int, default=5000, help="Quantidade de produtos (default: 5000)")
    ap.add_argument("--customers", type=int, default=2000, help="Quantidade de clientes (default: 2000)")
    ap.add_argument("--sales", type=int, default=10000, help="Quantidade de vendas (default: 10000)")
    ap.add_argument("--max-items", type=int, default=5, help="Máximo de itens por venda (default: 5)")
    ap.add_argument("--fast", action="store_true", help="Inserção rápida (PRAGMAs agressivos; uso local)")
    args = ap.parse_args()

    if args.db:
        os.environ["ERP_PDV_DB"] = args.db

    # schema base do app + extras
    init_db()
    ensure_extra_schema()

    if args.fast:
        with get_conn() as conn:
            apply_fast_pragmas(conn)

    seed_products(args.products)
    seed_customers(args.customers)  # necessário para vincular vendas
    if args.fast:
        with get_conn() as conn:
            apply_fast_pragmas(conn)
    seed_sales(args.sales, max_items=args.max_items)

    print("\n[done] carga concluída.")
    print("DB:", os.environ.get("ERP_PDV_DB") or get_db_path())

if __name__ == "__main__":
    main()
