﻿import argparse, datetime as dt, sqlite3, sys, uuid

DB_PATH = 'erp_pdv.db'
SALES_SOURCES = [("vendas","created_at"), ("sales","ts")]
POSSIBLE_DATE_COLUMNS = ["created_at","data_hora","datetime","data","timestamp","ts"]

def table_exists(cur, name):
    return cur.execute("SELECT 1 FROM sqlite_master WHERE type='table' AND name=? LIMIT 1",(name,)).fetchone() is not None

def pick_existing_sources(cur):
    sources = []
    for t, default_col in SALES_SOURCES:
        if table_exists(cur, t):
            cols = [c[1] for c in cur.execute(f"PRAGMA table_info('{t}')").fetchall()]
            col = default_col if default_col in cols else None
            if not col:
                for c in POSSIBLE_DATE_COLUMNS:
                    if c in cols:
                        col = c; break
            if col:
                sources.append((t,col))
    return sources

def get_cutoff_and_value(cur, arg_hora, arg_valor_cent):
    cutoff = None
    if table_exists(cur, 'taxas_config'):
        cols = [c[1] for c in cur.execute("PRAGMA table_info('taxas_config')").fetchall()]
        if 'hora_cobranca' in cols:
            row = cur.execute("SELECT hora_cobranca FROM taxas_config LIMIT 1").fetchone()
            if row and row[0]:
                cutoff = row[0]
    if not cutoff: cutoff = '22:00'
    if arg_hora:   cutoff = arg_hora
    valor_cents = 10 if arg_valor_cent is None else int(arg_valor_cent)
    return cutoff, round(valor_cents/100.0, 2)

def count_sales_until(cur, sources, date_ref_iso, cutoff_hhmm):
    total = 0
    for table, col in sources:
        q = f"SELECT COUNT(*) FROM {table} WHERE DATE({col}) = DATE(?) AND TIME({col}) <= ?"
        total += cur.execute(q, (date_ref_iso, cutoff_hhmm)).fetchone()[0]
    return total

def count_fees_today(cur, date_ref_iso):
    return cur.execute(
        "SELECT COUNT(*) FROM taxas_cobrancas WHERE DATE(created_at)=DATE(?)",
        (date_ref_iso,)
    ).fetchone()[0]

def backfill_empty_txid(cur):
    # Garante unicidade onde txid está NULL ou ''
    rows = cur.execute("SELECT id FROM taxas_cobrancas WHERE txid IS NULL OR txid = ''").fetchall()
    for (rid,) in rows:
        txid = f"FEE-{dt.datetime.now().strftime('%Y%m%d-%H%M%S')}-{uuid.uuid4().hex[:8]}"
        cur.execute("UPDATE taxas_cobrancas SET txid=? WHERE id=?", (txid, rid))

def insert_missing_fees(cur, missing, valor):
    now_iso = dt.datetime.now().isoformat(timespec='seconds')
    for _ in range(missing):
        txid = f"FEE-{dt.datetime.now().strftime('%Y%m%d-%H%M%S')}-{uuid.uuid4().hex[:8]}"
        # usa o mesmo txid como placeholder de copia_cola (até o provedor PIX devolver um real)
        cur.execute(
            "INSERT INTO taxas_cobrancas (created_at, valor, copia_cola, txid, status, quitadas_ate, amount) "
            "VALUES (?, ?, ?, ?, 'PENDENTE', NULL, ?)",
            (now_iso, valor, txid, txid, valor)
        )

def main():
    ap = argparse.ArgumentParser(description="Gera taxas do dia somando vendas de 'vendas' e 'sales'.")
    ap.add_argument("--hora", help="Hora de corte HH:MM (ex: 22:00). Se omitido, usa taxas_config.hora_cobranca ou 22:00.")
    ap.add_argument("--valor", type=int, help="Valor por venda em centavos (ex: 10 para R$0,10).")
    args = ap.parse_args()

    conn = sqlite3.connect(DB_PATH); cur = conn.cursor()

    if not table_exists(cur, 'taxas_cobrancas'):
        print("ERRO: tabela 'taxas_cobrancas' não encontrada."); sys.exit(2)

    sources = pick_existing_sources(cur)
    if not sources:
        print("ERRO: não encontrei tabela/coluna de data das vendas (vendas/sales)."); sys.exit(3)

    cutoff, valor = get_cutoff_and_value(cur, args.hora, args.valor)
    hoje = dt.date.today().isoformat()

    # normaliza txid vazio/nulo para evitar colisão em índice único
    backfill_empty_txid(cur)

    vendas = count_sales_until(cur, sources, hoje, cutoff)
    ja = count_fees_today(cur, hoje)

    faltam = max(0, vendas - ja)
    fontes_str = ", ".join([f"{t}.{c}" for t,c in sources])
    print(f"[INFO] Vendas ({fontes_str}) hoje até {cutoff}: {vendas} | taxas existentes hoje: {ja} | faltam gerar: {faltam}")

    if faltam > 0:
        insert_missing_fees(cur, faltam, valor)
        conn.commit()
        print(f"[OK] Geradas {faltam} taxa(s) PENDENTE de R$ {valor:.2f} (total R$ {faltam*valor:.2f}).")
    else:
        conn.commit()
        print("[OK] Nada a gerar — já está coerente com as vendas.")

    conn.close()

if __name__ == "__main__":
    main()
