from __future__ import annotations
import sqlite3
import threading
import json
import urllib.parse
import os
import sys
import tempfile

from datetime import datetime

from http.server import BaseHTTPRequestHandler, HTTPServer
import tkinter as tk
from tkinter import ttk, messagebox, filedialog
from typing import Callable, Optional, List, Tuple

# ========= HTTP para PSPs (polling) =========
try:
    import requests
    _REQ_OK = True
except Exception:
    _REQ_OK = False

# ========= Bloqueio por Taxas (opcional) =========
try:
    from fees_block import guard_block as _fees_guard_block  # retorna True se deve bloquear
except Exception:
    def _fees_guard_block(db_path: str, parent_window=None) -> bool:
        return False

# ========= Integração de Taxas (R$0,10 por venda) =========
# Requer o arquivo fees_utils.py na raiz do projeto
try:
    from fees_utils import ensure_tax_table, insert_fee, mark_fee_paid_by_sale
    _FEES_OK = True
except Exception:
    _FEES_OK = False

    def ensure_tax_table(conn):
        pass

    def insert_fee(conn, *, sale_id: int, status: str, amount: float = 0.10, txid: Optional[str] = None, copia_cola: str = ""):
        pass

    def mark_fee_paid_by_sale(conn, sale_id: int, *, payment_id: Optional[str] = None) -> int:
        return 0

# ========= Helpers moeda/pagamento =========
def normalize_payment_method(pm: str) -> str:
    s = (pm or "").strip().lower()
    if s in ("cash", "dinheiro", "money"):
        return "cash"
    if s in ("card", "cartao", "cartão", "credito", "crédito", "debito", "débito"):
        return "card"
    if s == "pix":
        return "pix"
    return "other"

def _money_to_float(s: str) -> float:
    if not s:
        return 0.0
    s = s.strip().replace("R$", "").strip()
    s = s.replace(".", "").replace(",", ".")
    try:
        return float(s)
    except Exception:
        return 0.0

def _fmt_money(v: float) -> str:
    return f"{v:,.2f}".replace(",", "X").replace(".", ",").replace("X", ".")

# ========= DB util =========
def _table_columns(conn: sqlite3.Connection, table: str) -> List[str]:
    try:
        return [r[1].lower() for r in conn.execute('PRAGMA table_info("{0}")'.format(table))]
    except Exception:
        return []

def ensure_sales_schema(conn: sqlite3.Connection) -> None:
    conn.execute("""
        CREATE TABLE IF NOT EXISTS sales (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            customer_id INTEGER,
            ts TEXT NOT NULL,
            total_gross REAL NOT NULL,
            discount_pct REAL NOT NULL DEFAULT 0,
            payment_method TEXT NOT NULL,
            received REAL NOT NULL DEFAULT 0,
            change REAL NOT NULL DEFAULT 0
        )
    """)
    conn.execute("""
        CREATE TABLE IF NOT EXISTS sale_items (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            sale_id INTEGER NOT NULL,
            product_id INTEGER NOT NULL,
            qty INTEGER NOT NULL,
            unit_price REAL NOT NULL,
            FOREIGN KEY(sale_id) REFERENCES sales(id) ON DELETE CASCADE
        )
    """)
    cols = _table_columns(conn, "sales")
    expected = {
        "customer_id": "ALTER TABLE sales ADD COLUMN customer_id INTEGER",
        "ts": "ALTER TABLE sales ADD COLUMN ts TEXT NOT NULL DEFAULT ''",
        "total_gross": "ALTER TABLE sales ADD COLUMN total_gross REAL NOT NULL DEFAULT 0",
        "discount_pct": "ALTER TABLE sales ADD COLUMN discount_pct REAL NOT NULL DEFAULT 0",
        "payment_method": "ALTER TABLE sales ADD COLUMN payment_method TEXT NOT NULL DEFAULT 'cash'",
        "received": "ALTER TABLE sales ADD COLUMN received REAL NOT NULL DEFAULT 0",
        "change": "ALTER TABLE sales ADD COLUMN change REAL NOT NULL DEFAULT 0",
        "buyer_cpf": "ALTER TABLE sales ADD COLUMN buyer_cpf TEXT"  # novo
        # 'status' é opcional para retrocompatibilidade
    }
    for c, ddl in expected.items():
        if c not in cols:
            conn.execute(ddl)

def _sales_has_status(conn: sqlite3.Connection) -> bool:
    return "status" in _table_columns(conn, "sales")

def _sales_has_buyer_cpf(conn: sqlite3.Connection) -> bool:
    return "buyer_cpf" in _table_columns(conn, "sales")

def _product_has_barcode(conn: sqlite3.Connection) -> bool:
    return "barcode" in _table_columns(conn, "products")

def _product_has_stock(conn: sqlite3.Connection) -> bool:
    return "stock_qty" in _table_columns(conn, "products")

def find_products(conn: sqlite3.Connection, q: str, limit: int = 30) -> List[Tuple[int,str,str,float]]:
    has_bar = _product_has_barcode(conn)
    q_norm = (q or "").strip()
    try:
        if not q_norm:
            cur = conn.execute("SELECT id, sku, name, unit_price FROM products ORDER BY name LIMIT ?", (int(limit),))
            return [(r[0], r[1], r[2], float(r[3] or 0.0)) for r in cur.fetchall()]
        if has_bar:
            row = conn.execute(
                "SELECT id, sku, name, unit_price FROM products WHERE sku=? OR barcode=? LIMIT 1", (q_norm, q_norm)
            ).fetchone()
        else:
            row = conn.execute("SELECT id, sku, name, unit_price FROM products WHERE sku=? LIMIT 1", (q_norm,)).fetchone()
        if row:
            return [(row[0], row[1], row[2], float(row[3] or 0.0))]
        like = "%{0}%".format(q_norm)
        if has_bar:
            cur = conn.execute(
                """SELECT id, sku, name, unit_price FROM products
                   WHERE name LIKE ? OR sku LIKE ? OR barcode LIKE ?
                   ORDER BY name LIMIT ?""",
                (like, like, like, int(limit)),
            )
        else:
            cur = conn.execute(
                """SELECT id, sku, name, unit_price FROM products
                   WHERE name LIKE ? OR sku LIKE ?
                   ORDER BY name LIMIT ?""",
                (like, like, int(limit)),
            )
        return [(r[0], r[1], r[2], float(r[3] or 0.0)) for r in cur.fetchall()]
    except sqlite3.Error as e:
        messagebox.showerror(
            "Produtos",
            "Não foi possível consultar os produtos.\n"
            "Verifique se a tabela 'products' existe (id, sku, name, unit_price [, barcode], [, stock_qty]).\n\n"
            "{0}".format(e),
            parent=_SAFE_PARENT()
        )
        return []

def decrement_stock_if_available(conn: sqlite3.Connection, product_id: int, qty: int) -> None:
    if not _product_has_stock(conn):
        return
    conn.execute("UPDATE products SET stock_qty = COALESCE(stock_qty,0) - ? WHERE id = ?", (int(qty), int(product_id)))

# ========= Webcam scanner (QR + barras) =========
_CV2_OK = True
_PIL_OK = True
_PYZBAR_OK = True
try:
    import cv2
except Exception:
    _CV2_OK = False
try:
    from PIL import Image, ImageTk
except Exception:
    _PIL_OK = False
try:
    from pyzbar import pyzbar
except Exception:
    _PYZBAR_OK = False

class WebcamScanner:
    def __init__(self, master: tk.Misc, on_code: Callable[[str], None]):
        self.master = master
        self.on_code = on_code
        self.cap = None
        self.running = False
        self.detector = None

        self.win = tk.Toplevel(master)
        self.win.title("Leitor — QR / Código de Barras (Webcam)")
        self.win.geometry("780x560")
        self.win.transient(master)
        self.win.grab_set()
        self.win.lift()
        try:
            self.win.attributes("-topmost", True)
        except Exception:
            pass
        self.win.protocol("WM_DELETE_WINDOW", self.close)
        self.win.bind("<Escape>", lambda e: self.close())

        info = ttk.Frame(self.win, padding=6)
        info.pack(fill="x")
        lbl = "Aponte o código para a câmera. Fecha com ESC."
        if not _PYZBAR_OK:
            lbl += " (pyzbar ausente: suporte limitado a QR Code)"
        ttk.Label(info, text=lbl).pack(side="left")
        ttk.Button(info, text="Fechar", command=self.close).pack(side="right")

        self.video = ttk.Label(self.win)
        self.video.pack(fill="both", expand=True, padx=6, pady=6)

        missing = []
        if not _CV2_OK:  missing.append("opencv-python")
        if not _PIL_OK:  missing.append("pillow")
        if missing:
            messagebox.showerror("Webcam", "Para usar o leitor instale:\n  pip install " + " ".join(missing),
                                 parent=master)
            self.win.after(100, self.close)
            return

        self.cap = self._open_camera()
        if self.cap is None:
            messagebox.showerror("Webcam", "Não foi possível abrir a câmera.", parent=master)
            self.win.after(100, self.close)
            return

        try:
            self.cap.set(cv2.CAP_PROP_FRAME_WIDTH, 1280)
            self.cap.set(cv2.CAP_PROP_FRAME_HEIGHT, 720)
        except Exception:
            pass

        try:
            self.detector = cv2.QRCodeDetector()
        except Exception:
            self.detector = None

        self.running = True
        self._loop()

    def _open_camera(self):
        if not _CV2_OK:
            return None
        indices = [0, 1, 2, 3]
        backends = []
        if hasattr(cv2, "CAP_DSHOW"): backends.append(cv2.CAP_DSHOW)
        if hasattr(cv2, "CAP_MSMF"):  backends.append(cv2.CAP_MSMF)
        backends.append(0)
        for be in backends:
            for idx in indices:
                cap = None
                try:
                    cap = cv2.VideoCapture(idx, be) if be != 0 else cv2.VideoCapture(idx)
                    if cap and cap.isOpened():
                        return cap
                finally:
                    if cap and not cap.isOpened():
                        cap.release()
        return None

    def _decode_codes(self, frame) -> Optional[str]:
        if _PYZBAR_OK:
            try:
                barcodes = pyzbar.decode(frame)
                for b in barcodes:
                    data = b.data.decode("utf-8", errors="ignore").strip()
                    if data:
                        return data
            except Exception:
                pass
        if self.detector is not None:
            try:
                data, _pts, _ = self.detector.detectAndDecode(frame)
                if data:
                    return data.strip()
            except Exception:
                pass
        return None

    def _loop(self):
        if not self.running or not self.cap:
            return
        ok, frame = self.cap.read()
        if ok:
            try:
                frame = cv2.flip(frame, 1)
            except Exception:
                pass
            code = self._decode_codes(frame)
            try:
                frame_rgb = cv2.cvtColor(frame, cv2.COLOR_BGR2RGB)
            except Exception:
                frame_rgb = frame
            if _PIL_OK:
                im = Image.fromarray(frame_rgb)
                im = im.resize((760, 520))
                imgtk = ImageTk.PhotoImage(image=im)
                self.video.imgtk = imgtk
                self.video.configure(image=imgtk)
            if code:
                try:
                    self.on_code(code)
                finally:
                    self.close()
                    return
        self.win.after(20, self._loop)

    def close(self):
        self.running = False
        try:
            if self.cap: self.cap.release()
        except Exception:
            pass
        try:
            self.win.destroy()
        except Exception:
            pass

# ========= PIX: settings + BR Code (QR) =========
_QR_OK = True
try:
    import qrcode
    from PIL import Image, ImageTk  # para o QR renderizado
except Exception:
    _QR_OK = False

def _ensure_settings(conn: sqlite3.Connection):
    conn.execute("""
        CREATE TABLE IF NOT EXISTS settings (
            key   TEXT PRIMARY KEY,
            value TEXT
        )
    """)

def _get_setting(conn: sqlite3.Connection, key: str, default: str = "") -> str:
    _ensure_settings(conn)
    row = conn.execute("SELECT value FROM settings WHERE key=?", (key,)).fetchone()
    return row[0] if row and row[0] is not None else default

# >>> NOVO: nome do estabelecimento a partir das configs
def _get_business_name(conn) -> str:
    for key in ("NFCE_FANTASIA", "NFCE_EMITENTE_RAZAO", "BUSINESS_NAME", "COMPANY_NAME", "STORE_NAME"):
        v = _get_setting(conn, key, "").strip()
        if v:
            return v
    v = _get_setting(conn, "PIX_MERCHANT_NAME", "").strip()
    return v or "LOJA"

def _emv(tag: str, value: str) -> str:
    return "{0}{1:02d}{2}".format(tag, len(value), value)

def _crc16(payload: str) -> str:
    poly = 0x1021
    crc = 0xFFFF
    data = (payload + "6304").encode("utf-8")
    for b in data:
        crc ^= (b << 8)
        for _ in range(8):
            if (crc & 0x8000):
                crc = ((crc << 1) ^ poly) & 0xFFFF
            else:
                crc = (crc << 1) & 0xFFFF
    return "{0:04X}".format(crc)

def build_pix_payload(*, chave: str, nome: str, cidade: str, valor: float, txid: str, desc: str = "") -> str:
    nome   = (nome or "LOJA")[:25]
    cidade = (cidade or "SAO PAULO").upper()[:15]
    txid   = (txid or "ERP")[:25]
    p00 = _emv("00", "01")
    p01 = _emv("01", "11")
    mai = _emv("00", "br.gov.bcb.pix") + _emv("01", chave)
    if desc:
        mai += _emv("02", desc[:50])
    p26 = _emv("26", mai)
    p52 = _emv("52", "0000")
    p53 = _emv("53", "986")
    p54 = _emv("54", "{0:.2f}".format(valor).replace(",", "."))
    p58 = _emv("58", "BR")
    p59 = _emv("59", nome)
    p60 = _emv("60", cidade)
    p62 = _emv("62", _emv("05", txid))
    partial = p00 + p01 + p26 + p52 + p53 + p54 + p58 + p59 + p60 + p62
    return partial + _emv("63", _crc16(partial))

def _random_txid(prefix="FD") -> str:
    import random, string
    return (prefix + "".join(random.choice(string.ascii_uppercase + string.digits) for _ in range(8)))[:25]

# ========= Servidor Webhook (opcional) =========
_WEBHOOK_SERVER = None

class _WebhookHandler(BaseHTTPRequestHandler):
    def do_POST(self):
        token_hdr = self.headers.get('X-Auth-Token')
        qs = urllib.parse.urlparse(self.path).query
        q = urllib.parse.parse_qs(qs)
        token_qs = q.get('token', [None])[0]
        token = getattr(self.server, "_token", None)  # type: ignore
        if token and token not in (token_hdr, token_qs):
            self._send(403, b"forbidden")
            return

        length = int(self.headers.get('Content-Length', '0') or 0)
        raw = self.rfile.read(length) if length > 0 else b'{}'
        try:
            data = json.loads(raw.decode('utf-8') or '{}')
        except Exception:
            data = {}

        payment_id = None
        txid = None
        if isinstance(data, dict):
            if "data" in data and isinstance(data["data"], dict):
                payment_id = str(data["data"].get("id") or "")
            if not payment_id and "id" in data:
                payment_id = str(data.get("id") or "")
            if "txid" in data:
                txid = str(data.get("txid") or "")

        if not payment_id and "payment_id" in q:
            payment_id = q["payment_id"][0]

        subs = getattr(self.server, "_subs", [])  # type: ignore
        for cb in list(subs):
            try:
                cb({"payment_id": payment_id, "txid": txid, "raw": data})
            except Exception:
                pass

        self._send(200, b"ok")

    def log_message(self, format, *args):
        return

    def _send(self, code: int, body: bytes):
        self.send_response(code)
        self.end_headers()
        try:
            self.wfile.write(body)
        except Exception:
            pass

class PixWebhookServer:
    def __init__(self, port: int, token: str = ""):
        self.port = int(port)
        self.token = token or ""
        self.httpd = HTTPServer(("0.0.0.0", self.port), _WebhookHandler)
        self.httpd._subs = []          # type: ignore
        self.httpd._token = self.token # type: ignore
        self.thread = threading.Thread(target=self.httpd.serve_forever, daemon=True)

    def start(self):
        if not self.thread.is_alive():
            self.thread.start()

    def subscribe(self, cb: Callable[[dict], None]):
        self.httpd._subs.append(cb)    # type: ignore

# ========= UI: Janela PDV =========
def _SAFE_PARENT():
    # helper para messagebox mesmo antes da classe estar instanciada
    try:
        # Se existir uma instância, devolvo a janela do PDV ativa
        return tk._default_root.focus_get().winfo_toplevel()
    except Exception:
        return None

class PDVWindow:
    def __init__(self, app, get_conn: Callable):
        self.app = app
        self.get_conn = get_conn

        self.root = tk.Toplevel(app.root)
        self.root.title("PDV (Vendas)")
        try:
            self.root.state('zoomed')  # abre maximizada
        except Exception:
            self.root.geometry("980x720")

        # >>> Sempre em evidência
        try:
            self.root.attributes("-topmost", True)
        except Exception:
            pass
        self.root.lift()
        self.root.focus_force()

        # Estado
        self.cart: List[dict] = []
        self.var_search = tk.StringVar()
        self.var_qty = tk.IntVar(value=1)
        self.var_discount = tk.DoubleVar(value=0.0)
        self.var_total = tk.StringVar(value="R$ 0,00")
        self.var_subtotal = tk.StringVar(value="R$ 0,00")
        self.var_pm = tk.StringVar(value="cash")
        self.var_received = tk.StringVar(value="R$ 0,00")
        self.var_change = tk.StringVar(value="R$ 0,00")

        # PIX
        self._pix_payload = ""
        self._qr_imgtk: Optional['ImageTk.PhotoImage'] = None if _QR_OK else None
        self._pix_provider = None
        self._pix_provider_id = None
        self._pix_poll_job = None

        # ID de venda pendente (para atualizar status quando PIX aprovar)
        self._pending_sale_id: Optional[int] = None

        # Topo: busca
        top = ttk.Frame(self.root, padding=8)
        top.pack(fill="x")
        ttk.Label(top, text="Pesquisar (SKU/Barra/Nome):").pack(side="left")
        ent_search = ttk.Entry(top, textvariable=self.var_search, width=40)
        ent_search.pack(side="left", padx=(6,6))
        ttk.Label(top, text="Qtd:").pack(side="left")
        spn = ttk.Spinbox(top, from_=1, to=9999, textvariable=self.var_qty, width=6)
        spn.pack(side="left", padx=(6,6))
        ttk.Button(top, text="Buscar / Adicionar (Enter)", command=self._search_and_add).pack(side="left")
        ttk.Button(top, text="📷 Ler QR/Código (F7)", command=self._open_scanner).pack(side="left", padx=(8,0))

        # Layout principal
        body = ttk.Frame(self.root, padding=(8,0,8,8)); body.pack(fill="both", expand=True)
        left = ttk.Frame(body); left.pack(side="left", fill="both", expand=True)
        right = ttk.Frame(body); right.pack(side="right", fill="y")

        ttk.Label(left, text="Itens da venda", font=("Segoe UI", 11, "bold")).pack(anchor="w")
        self.tv_cart = ttk.Treeview(left, columns=("sku","name","qty","unit","total"), show="headings", height=14)
        for c,t,w,a in [("sku","SKU",120,"w"),("name","Produto",320,"w"),("qty","Qtd",60,"e"),
                        ("unit","Unit (R$)",90,"e"),("total","Total (R$)",100,"e")]:
            self.tv_cart.heading(c, text=t); self.tv_cart.column(c, width=w, anchor=a)
        self.tv_cart.pack(fill="both", expand=True, pady=(4,8))

        ttk.Label(left, text="Resultados da busca (duplo clique para adicionar)", font=("Segoe UI", 10, "bold")).pack(anchor="w")
        self.tv_search = ttk.Treeview(left, columns=("sku","name","unit"), show="headings", height=10)
        for c,t,w,a in [("sku","SKU",120,"w"),("name","Produto",320,"w"),("unit","Unit (R$)",100,"e")]:
            self.tv_search.heading(c, text=t); self.tv_search.column(c, width=w, anchor=a)
        self.tv_search.pack(fill="both", expand=True, pady=(4,0))
        self.tv_search.bind("<Double-1>", lambda e: self._add_selected_search())

        box = ttk.LabelFrame(right, text="Totais e Pagamento", padding=8); box.pack(fill="y", side="top", padx=(8,0), pady=(0,8))
        row = 0
        ttk.Label(box, text="Subtotal:").grid(row=row, column=0, sticky="e")
        ttk.Label(box, textvariable=self.var_subtotal, font=("Segoe UI", 10, "bold")).grid(row=row, column=1, sticky="w")
        row += 1

        ttk.Label(box, text="Desconto (%):").grid(row=row, column=0, sticky="e")
        ent_disc = ttk.Entry(box, textvariable=self.var_discount, width=8)
        ent_disc.grid(row=row, column=1, sticky="w")
        row += 1
        ent_disc.bind("<KeyRelease>", lambda e: self._recalc_totals())

        ttk.Label(box, text="Total:").grid(row=row, column=0, sticky="e")
        ttk.Label(box, textvariable=self.var_total, font=("Segoe UI", 12, "bold")).grid(row=row, column=1, sticky="w")
        row += 1

        ttk.Label(box, text="Pagamento:").grid(row=row, column=0, sticky="e")
        row += 1
        frm_pm = ttk.Frame(box)
        frm_pm.grid(row=row, column=0, columnspan=2, sticky="w")
        row += 1
        r1 = ttk.Radiobutton(frm_pm, text="Dinheiro", value="cash", variable=self.var_pm, command=self._on_pm_change)
        r2 = ttk.Radiobutton(frm_pm, text="PIX",      value="pix",  variable=self.var_pm, command=self._on_pm_change)
        r3 = ttk.Radiobutton(frm_pm, text="Cartão",   value="card", variable=self.var_pm, command=self._on_pm_change)
        r1.pack(side="left"); r2.pack(side="left", padx=8); r3.pack(side="left")

        ttk.Label(box, text="Recebido:").grid(row=row, column=0, sticky="e")
        self.ent_received = ttk.Entry(box, textvariable=self.var_received, width=14)
        self.ent_received.grid(row=row, column=1, sticky="w")
        row += 1
        self.ent_received.bind("<KeyRelease>", lambda e: self._recalc_change())

        ttk.Label(box, text="Troco:").grid(row=row, column=0, sticky="e")
        ttk.Label(box, textvariable=self.var_change, font=("Segoe UI", 11, "bold")).grid(row=row, column=1, sticky="w")
        row += 1

        # Botão Finalizar (desabilitado no PIX)
        self.btn_finalize = ttk.Button(box, text="Finalizar venda (F12)", command=self.finalize_sale)
        self.btn_finalize.grid(row=row, column=0, columnspan=2, pady=(8,0))
        row += 1

        ttk.Button(box, text="Cancelar / Limpar", command=self._clear_cart).grid(row=row, column=0, columnspan=2, pady=(6,0))
        row += 1
        ttk.Button(box, text="Confirmar pagamento", command=self._confirm_payment_dialog).grid(row=row, column=0, columnspan=2, pady=(6,0))
        row += 1

        # PIX (QR + copia e cola)
        self.frm_pix = ttk.LabelFrame(right, text="PIX — QR / Copia e Cola", padding=8)
        self.frm_pix.pack(fill="x", side="top", padx=(8,0), pady=(0,8))
        self.lbl_qr = ttk.Label(self.frm_pix); self.lbl_qr.pack(pady=(0,6))
        self.txt_copia = tk.Text(self.frm_pix, height=4, width=34, wrap="word")
        self.txt_copia.configure(state="disabled"); self.txt_copia.pack(fill="x")
        btns = ttk.Frame(self.frm_pix); btns.pack(pady=6)
        ttk.Button(btns, text="Copiar código", command=self._copy_copia).pack(side="left", padx=4)
        ttk.Button(btns, text="Salvar QR (PNG)", command=self._save_qr_png).pack(side="left", padx=4)
        self.lbl_pix_status = ttk.Label(self.frm_pix, text="", foreground="#0a7"); self.lbl_pix_status.pack()
        self.lbl_pix_id = ttk.Label(self.frm_pix, text="", foreground="#666"); self.lbl_pix_id.pack()
        # Gera a cobrança PIX (cria venda PENDENTE + QR)
        self.btn_pix_charge = ttk.Button(self.frm_pix, text="Gerar cobrança PIX", command=self._create_pix_charge)
        self.btn_pix_charge.pack(pady=(6,0))
        ttk.Button(self.frm_pix, text="Reconsultar agora", command=self._poll_once).pack(pady=(4,0))

        # binds
        self.root.bind("<Return>", lambda e: self._search_and_add())
        self.root.bind("<F12>", lambda e: self._handle_f12())
        self.root.bind("<F7>",  lambda e: self._open_scanner())

        with self.get_conn() as conn:
            ensure_sales_schema(conn)
            _ensure_settings(conn)

        self._recalc_totals()
        self._on_pm_change()

    # ---------- Helpers de foco/posição ----------
    def _bring_pdv_front(self):
        try:
            self.root.lift()
            self.root.focus_force()
            self.root.after(10, lambda: self.root.focus_force())
        except Exception:
            pass

    def _center_on_parent(self, win: tk.Toplevel, parent: tk.Toplevel):
        try:
            parent.update_idletasks()
            pw = parent.winfo_width()
            ph = parent.winfo_height()
            px = parent.winfo_rootx()
            py = parent.winfo_rooty()
            win.update_idletasks()
            ww = win.winfo_width()
            wh = win.winfo_height()
            x = px + (pw - ww) // 2
            y = py + (ph - wh) // 2
            win.geometry(f"+{max(x,0)}+{max(y,0)}")
        except Exception:
            pass

    def _make_modal(self, win: tk.Toplevel):
        try:
            win.transient(self.root)
            win.grab_set()
            win.lift()
            try:
                win.attributes("-topmost", True)
            except Exception:
                pass
            self._center_on_parent(win, self.root)
        except Exception:
            pass

    # --- util estado botão finalizar
    def _update_finalize_state(self):
        is_pix = (self.var_pm.get() == "pix")
        try:
            self.btn_finalize.config(state=("disabled" if is_pix else "normal"))
        except Exception:
            pass

    def _handle_f12(self):
        if self.var_pm.get() == "pix":
            messagebox.showwarning("PDV", "Para PIX, aguarde a confirmação do pagamento. O PDV finalizará automaticamente.",
                                   parent=self.root)
            self._bring_pdv_front()
            return "break"
        self.finalize_sale()
        return "break"

    # ----- Settings helpers -----
    def _read_pix_settings(self):
        with self.get_conn() as conn:
            prov   = _get_setting(conn, "PIX_PROVIDER", "simulado").strip().lower()
            base   = _get_setting(conn, "PIX_API_BASE", "").strip()
            token  = _get_setting(conn, "PIX_CLIENT_SECRET", "").strip()
            poll_s = _get_setting(conn, "PIX_POLL_INTERVAL", "3").strip()
            mode   = _get_setting(conn, "PIX_CONFIRM_MODE", "none").strip().lower()
            wport  = _get_setting(conn, "PIX_WEBHOOK_PORT", "5001").strip()
            wtok   = _get_setting(conn, "PIX_WEBHOOK_TOKEN", "").strip()
            wurl   = _get_setting(conn, "PIX_WEBHOOK_URL", "").strip()
        try:
            poll_s = max(2, int(poll_s))
        except Exception:
            poll_s = 3
        try:
            wport = int(wport)
        except Exception:
            wport = 5001
        if not base and prov == "mercadopago":
            base = "https://api.mercadopago.com"
        return prov, base, token, poll_s, mode, wport, wtok, wurl

    # ----- Webcam -----
    def _open_scanner(self):
        # Usa o PDV como pai (fica na frente dele)
        WebcamScanner(self.root, lambda code: self._handle_scanned_code(code))

    def _handle_scanned_code(self, code: str):
        code = (code or "").strip()
        if not code:
            return
        self.var_search.set(code)
        with self.get_conn() as conn:
            rows = find_products(conn, code, limit=1)
        if rows:
            qty = max(1, int(self.var_qty.get() or 1))
            self._add_product(rows[0], qty)
            self.var_search.set("")
        else:
            messagebox.showwarning("Leitor", "Código '{0}' não cadastrado.".format(code), parent=self.root)
        self._refresh_search_list(code)
        self._bring_pdv_front()

    # ----- Busca / Carrinho -----
    def _refresh_search_list(self, q: str):
        with self.get_conn() as conn:
            rows = find_products(conn, q, limit=40)
        for iid in self.tv_search.get_children():
            self.tv_search.delete(iid)
        for (pid, sku, name, unit) in rows:
            self.tv_search.insert("", "end", iid=str(pid), values=(sku, name, _fmt_money(unit)))

    def _search_and_add(self):
        q = self.var_search.get().strip()
        qty = max(1, int(self.var_qty.get() or 1))
        with self.get_conn() as conn:
            rows = find_products(conn, q, limit=40)
        for iid in self.tv_search.get_children():
            self.tv_search.delete(iid)
        for (pid, sku, name, unit) in rows:
            self.tv_search.insert("", "end", iid=str(pid), values=(sku, name, _fmt_money(unit)))
        if len(rows) == 1 and q:
            self._add_product(rows[0], qty); self.var_search.set("")
        elif len(rows) == 0 and q:
            messagebox.showwarning("Busca", "Nenhum produto encontrado.", parent=self.root)
            self._bring_pdv_front()

    def _add_selected_search(self):
        qty = max(1, int(self.var_qty.get() or 1))
        sel = self.tv_search.selection()
        if not sel:
            return
        pid = int(sel[0])
        sku, name, unit_str = self.tv_search.item(sel[0], "values")
        unit = _money_to_float(unit_str)
        self._add_product((pid, sku, name, unit), qty)

    def _add_product(self, row: Tuple[int,str,str,float], qty: int):
        pid, sku, name, unit = row
        for it in self.cart:
            if it["product_id"] == pid:
                it["qty"] += qty
                self._refresh_cart_tv(); self._recalc_totals(); return
        self.cart.append({"product_id": pid, "sku": sku, "name": name, "unit_price": float(unit), "qty": int(qty)})
        self._refresh_cart_tv(); self._recalc_totals()

    def _refresh_cart_tv(self):
        for iid in self.tv_cart.get_children():
            self.tv_cart.delete(iid)
        for i, it in enumerate(self.cart, start=1):
            total = it["unit_price"] * it["qty"]
            self.tv_cart.insert("", "end", iid=str(i),
                                values=(it["sku"], it["name"], it["qty"], _fmt_money(it["unit_price"]), _fmt_money(total)))

    # ----- Totais / Pagamento -----
    def _cart_subtotal(self) -> float:
        return sum(it["unit_price"] * it["qty"] for it in self.cart)

    def _recalc_totals(self):
        subtotal = self._cart_subtotal()
        disc = float(self.var_discount.get() or 0.0); disc = max(0.0, min(disc, 100.0))
        total = subtotal * (1.0 - disc/100.0)
        self.var_subtotal.set("R$ " + _fmt_money(subtotal))
        self.var_total.set("R$ " + _fmt_money(total))
        self.var_received.set("R$ " + _fmt_money(total))
        self._recalc_change()

    def _on_pm_change(self):
        total = _money_to_float(self.var_total.get())
        self.var_received.set("R$ " + _fmt_money(total))
        self._recalc_change()
        if self.var_pm.get() in ("card", "pix"):
            self.ent_received.configure(state="disabled")
        else:
            self.ent_received.configure(state="normal")
        self._toggle_pix_area()
        if self.var_pm.get() != "pix":
            self._stop_pix_polling()
            self._pix_provider_id = None
            self.lbl_pix_id.config(text="")
        self._update_finalize_state()

    def _recalc_change(self):
        total = _money_to_float(self.var_total.get())
        received = _money_to_float(self.var_received.get())
        pm = self.var_pm.get()
        if pm == "cash":
            change = max(received - total, 0.0)
        else:
            received = total; change = 0.0
            self.var_received.set("R$ " + _fmt_money(received))
        self.var_change.set("R$ " + _fmt_money(change))

    def _clear_cart(self):
        self.cart.clear()
        self._refresh_cart_tv()
        self._recalc_totals()
        self._stop_pix_polling()
        self._pix_provider_id = None
        self.lbl_pix_id.config(text="")

    # ----- PIX UI -----
    def _toggle_pix_area(self):
        show = (self.var_pm.get() == "pix")
        if show and not self.frm_pix.winfo_ismapped():
            self.frm_pix.pack(fill="x", side="top", padx=(8,0), pady=(0,8))
        if not show and self.frm_pix.winfo_ismapped():
            self.frm_pix.pack_forget()

    def _set_copia_text(self, s: str, error: bool=False):
        self.txt_copia.configure(state="normal")
        self.txt_copia.delete("1.0", "end")
        self.txt_copia.insert("1.0", s)
        self.txt_copia.configure(state="disabled")
        self.lbl_pix_status.config(foreground=("#c00" if error else "#0a7"))

    def _copy_copia(self):
        if not self._pix_payload:
            return
        try:
            self.root.clipboard_clear()
            self.root.clipboard_append(self._pix_payload)
            messagebox.showinfo("PIX", "Código (copia e cola) copiado.", parent=self.root)
        except Exception:
            pass
        self._bring_pdv_front()

    def _save_qr_png(self):
        if not self._qr_imgtk or not self._pix_payload:
            return
        path = filedialog.asksaveasfilename(defaultextension=".png", initialfile="pix_qr.png",
                                            filetypes=[("PNG", "*.png")])
        if not path:
            return
        try:
            img = qrcode.make(self._pix_payload); img.save(path)
            messagebox.showinfo("PIX", "QR salvo em:\n{0}".format(path), parent=self.root)
        except Exception as e:
            messagebox.showerror("PIX", "Falha ao salvar PNG:\n{0}".format(e), parent=self.root)
        self._bring_pdv_front()

    # ----- PSP: Mercado Pago -----
    def _mp_create_charge(self, total: float) -> Optional[Tuple[str, str]]:
        prov, base, token, _poll_s, _mode, _wport, _wtok, wurl = self._read_pix_settings()
        if not _REQ_OK or not token or prov != "mercadopago":
            return None

        payer_email = "cliente@example.com"
        prefix = "FD"
        desc = "PDV"
        try:
            with self.get_conn() as conn:
                maybe_email = _get_setting(conn, "PIX_PAYER_EMAIL", "")
                if maybe_email:
                    payer_email = maybe_email.strip()
                prefix = _get_setting(conn, "PIX_TXID_PREFIX", "FD") or "FD"
                # >>> usa nome do negócio na descrição
                desc = f"PDV {_get_business_name(conn)}"
        except Exception:
            pass

        txid = _random_txid(prefix)
        idempotency_key = txid

        url = "{0}/v1/payments".format(base.rstrip('/'))
        headers = {
            "Authorization": "Bearer {0}".format(token),
            "Content-Type": "application/json",
            "X-Idempotency-Key": idempotency_key,
        }
        body = {
            "transaction_amount": float(total),
            "description": desc,  # antes: "PDV FabricaDigital.shop"
            "payment_method_id": "pix",
            "external_reference": txid,
            "payer": {"email": payer_email},
        }
        if wurl:
            body["notification_url"] = wurl

        try:
            r = requests.post(url, json=body, headers=headers, timeout=15)
            if r.status_code in (400, 409) and "Idempotency" in (r.text or ""):
                headers["X-Idempotency-Key"] = _random_txid(prefix)
                r = requests.post(url, json=body, headers=headers, timeout=15)

            if r.status_code >= 400:
                try:
                    err = r.json()
                except Exception:
                    err = {"error": r.text}
                self.lbl_pix_status.config(
                    text="Erro MP {0}: {1}".format(r.status_code, err.get('message') or err.get('error','')),
                    foreground="#c00"
                )
                print("[PIX/MP] create error", r.status_code, err)
                return None

            data = r.json()
            poi = data.get("point_of_interaction", {}).get("transaction_data", {})
            payload = poi.get("qr_code")
            payment_id = str(data.get("id"))
            if payload and payment_id:
                return payload, payment_id

        except Exception as e:
            self.lbl_pix_status.config(text="Falha ao criar cobrança: {0}".format(e), foreground="#c00")
            print("[PIX/MP] erro ao criar cobrança:", e)

        return None

    def _mp_check_paid(self, payment_id: str) -> bool:
        prov, base, token, *_ = self._read_pix_settings()
        if not _REQ_OK or not token or prov != "mercadopago" or not payment_id:
            return False
        url = "{0}/v1/payments/{1}".format(base.rstrip('/'), payment_id)
        headers = {"Authorization": "Bearer {0}".format(token)}
        try:
            r = requests.get(url, headers=headers, timeout=10)
            if r.status_code >= 400:
                print("[PIX/MP] status error", r.status_code, r.text[:300])
                return False
            data = r.json()
            status = (data.get("status") or "").lower()
            return status == "approved"
        except Exception as e:
            print("[PIX/MP] erro ao consultar status:", e)
            return False

    # ----- Polling -----
    def _start_pix_polling(self):
        self._stop_pix_polling()
        _prov, _base, _token, poll_s, mode, _wport, _wtok, _wurl = self._read_pix_settings()
        if mode in ("polling", "auto") and self._pix_provider_id:
            self._pix_poll_job = self.root.after(poll_s * 1000, self._poll_once)

    def _stop_pix_polling(self):
        if self._pix_poll_job:
            try:
                self.root.after_cancel(self._pix_poll_job)
            except Exception:
                pass
        self._pix_poll_job = None

    def _poll_once(self):
        paid = False
        if self._pix_provider == "mercadopago" and self._pix_provider_id:
            paid = self._mp_check_paid(self._pix_provider_id)
        if paid:
            self._on_pix_paid_confirmed("polling")
            return
        self._start_pix_polling()

    # ----- Webhook -----
    def _ensure_webhook_server(self):
        global _WEBHOOK_SERVER
        _prov, _base, _token, _poll_s, mode, wport, wtok, wurl = self._read_pix_settings()
        if mode not in ("webhook", "auto"):
            return
        if _WEBHOOK_SERVER is None:
            try:
                _WEBHOOK_SERVER = PixWebhookServer(wport, wtok)
                _WEBHOOK_SERVER.start()
            except Exception as e:
                print("[PIX] Falha ao iniciar webhook local:", e)
                return
        def _cb(payload: dict):
            payment_id = str(payload.get("payment_id") or "")
            txid = str(payload.get("txid") or "")
            self.root.after(0, lambda: self._handle_webhook_event(payment_id, txid))
        try:
            _WEBHOOK_SERVER.subscribe(_cb)
        except Exception:
            pass
        local_hint = "http://127.0.0.1:{0}/pix-webhook".format(wport)
        txt = "Aguardando pagamento via Webhook…"
        if wurl:
            txt += " Configure seu PSP para chamar:\n{0}".format(wurl)
        else:
            txt += " (endpoint local: {0})".format(local_hint)
        self.lbl_pix_status.config(text=txt, foreground="#444")

    def _handle_webhook_event(self, payment_id: str, _txid: str):
        if payment_id and self._pix_provider == "mercadopago":
            if self._pix_provider_id and payment_id != self._pix_provider_id:
                return
            if self._mp_check_paid(payment_id):
                self._on_pix_paid_confirmed("webhook")

    # ====== AUTO-FINALIZAR PIX ======
    def _on_pix_paid_confirmed(self, by: str):
        """Confirma venda PIX e finaliza automaticamente (taxa permanece PENDENTE) + impressão."""
        self.lbl_pix_status.config(text="Pagamento PIX confirmado ✔ ({0})".format(by), foreground="#0a7")
        self._stop_pix_polling()

        sale_id = self._pending_sale_id
        if sale_id:
            try:
                with self.get_conn() as conn:
                    if _sales_has_status(conn):
                        conn.execute("UPDATE sales SET status='PAGA' WHERE id=?", (sale_id,))
                    try:
                        conn.execute("""
                            UPDATE payments
                               SET status='PAID', updated_at=datetime('now')
                             WHERE sale_id=? AND status!='PAID'
                        """, (sale_id,))
                    except Exception:
                        pass
                    ensure_tax_table(conn)
            except Exception as e:
                print("[PDV] falha ao atualizar status da venda:", e)

        try:
            if sale_id:
                self._emit_nfce_or_receipt(sale_id)
        except Exception as e:
            print("[PDV] impressão pós-PIX falhou:", e)

        self._pending_sale_id = None
        self._pix_provider_id = None
        self._pix_payload = ""
        try:
            self.lbl_qr.configure(image="")
        except Exception:
            pass
        self.lbl_pix_id.config(text="")
        self._update_finalize_state()
        try:
            self.app.root.event_generate("<<SalePaid>>", when="tail")
        except Exception:
            pass

        messagebox.showinfo("PIX", "Pagamento confirmado ({0}). Venda finalizada.".format(by), parent=self.root)
        self._bring_pdv_front()

    # ----- Geração do QR -----
    def _create_pix_charge(self):
        if not self.cart:
            messagebox.showwarning("PIX", "Nenhum item no carrinho.", parent=self.root)
            self._bring_pdv_front()
            return

        # CPF opcional antes de criar a venda pendente (PIX)
        buyer_cpf = self._ask_buyer_cpf_optional()

        subtotal = self._cart_subtotal()
        discount_pct = float(self.var_discount.get() or 0.0)
        total_gross = subtotal * (1.0 - max(0.0, min(discount_pct, 100.0))/100.0)
        received = total_gross
        change = 0.0
        try:
            if self._pending_sale_id:
                messagebox.showinfo("PIX", "Cobrança PIX já criada para a venda #{0}. Aguardando confirmação…".format(self._pending_sale_id),
                                    parent=self.root)
                self._update_pix_qr()
                self._bring_pdv_front()
                return
            with self.get_conn() as conn:
                ensure_sales_schema(conn)
                sale_id = self._insert_sale(conn, status="PENDENTE", pm="pix",
                                            total_gross=total_gross, discount_pct=discount_pct,
                                            received=received, change=change, buyer_cpf=buyer_cpf)
                for it in self.cart:
                    conn.execute(
                        "INSERT INTO sale_items (sale_id, product_id, qty, unit_price) VALUES (?,?,?,?)",
                        (sale_id, it["product_id"], int(it["qty"]), float(it["unit_price"]))
                    )
                    decrement_stock_if_available(conn, it["product_id"], int(it["qty"]))
                try:
                    ensure_tax_table(conn)
                    insert_fee(conn, sale_id=sale_id, status="PENDENTE", amount=0.10)
                except Exception as e:
                    print("[TAXAS] falha ao registrar taxa:", e)
            self._pending_sale_id = sale_id
            self._update_pix_qr()
            messagebox.showinfo("PIX", "Cobrança PIX criada para a venda #{0}. Aguardando confirmação…".format(sale_id),
                                parent=self.root)
        except Exception as e:
            messagebox.showerror("PIX", "Falha ao criar cobrança PIX:\n{0}".format(e), parent=self.root)
        self._bring_pdv_front()

    def _update_pix_qr(self):
        if not _QR_OK:
            self.lbl_qr.configure(image=""); self._qr_imgtk = None
            self._set_copia_text("Instale para gerar QR: pip install qrcode[pil] pillow", error=True)
            self._pix_payload = ""
            return

        total = _money_to_float(self.var_total.get())
        if total <= 0:
            self.lbl_qr.configure(image=""); self._qr_imgtk = None
            self._set_copia_text("Total deve ser maior que 0 para gerar o PIX.")
            self._stop_pix_polling(); self._pix_provider_id = None
            self.lbl_pix_id.config(text="")
            return

        prov, _base, token, *_rest = self._read_pix_settings()
        self._pix_provider = prov
        self._pix_provider_id = None
        self._stop_pix_polling()

        payload = None

        # 1) PSP (Mercado Pago)
        if prov == "mercadopago" and token:
            ret = self._mp_create_charge(total)
            if ret:
                payload, payment_id = ret
                self._pix_provider_id = payment_id
                self.lbl_pix_status.config(text="Aguardando pagamento (Mercado Pago)…", foreground="#444")
                self.lbl_pix_id.config(text="ID do pagamento: {0}".format(payment_id))
                self._start_pix_polling()
                self._ensure_webhook_server()

        # 2) Fallback: QR estático local
        if not payload:
            with self.get_conn() as conn:
                chave  = _get_setting(conn, "PIX_KEY", "")
                nome   = _get_setting(conn, "PIX_MERCHANT_NAME", "")
                if not nome:
                    nome = _get_business_name(conn)  # <<< usa nome do negócio se PIX_MERCHANT_NAME não estiver setado
                cidade = _get_setting(conn, "PIX_MERCHANT_CITY", "SAO PAULO")
                prefix = _get_setting(conn, "PIX_TXID_PREFIX", "FD")
            if not chave:
                self.lbl_qr.configure(image=""); self._qr_imgtk = None
                self._set_copia_text("Chave PIX não configurada. Admin ▸ Configuração PIX.", error=True)
                self._stop_pix_polling(); self._pix_provider_id = None
                self.lbl_pix_id.config(text="")
                return
            txid = _random_txid(prefix)
            payload = build_pix_payload(chave=chave, nome=nome, cidade=cidade, valor=total, txid=txid, desc="PDV")
            self.lbl_pix_status.config(text="QR estático exibido (sem confirmação automática).", foreground="#444")
            self.lbl_pix_id.config(text="")

        # Renderizar QR e copia-e-cola
        try:
            img = qrcode.make(payload).resize((220, 220))
            self._qr_imgtk = ImageTk.PhotoImage(img)
            self.lbl_qr.configure(image=self._qr_imgtk)
            self._set_copia_text(payload)
            self._pix_payload = payload
        except Exception as e:
            self.lbl_qr.configure(image=""); self._qr_imgtk = None
            self._set_copia_text("Falha ao gerar QR: {0}".format(e), error=True)

    # ----- Gravação da venda -----
    def _insert_sale(self, conn: sqlite3.Connection, *, status: Optional[str], pm: str,
                     total_gross: float, discount_pct: float, received: float, change: float,
                     buyer_cpf: str = "") -> int:
        has_status = _sales_has_status(conn)
        has_bcpf = _sales_has_buyer_cpf(conn)

        if has_status and has_bcpf:
            cur = conn.execute(
                """INSERT INTO sales (customer_id, ts, total_gross, discount_pct, payment_method, received, "change", status, buyer_cpf)
                   VALUES (NULL, datetime('now','localtime'), ?, ?, ?, ?, ?, ?, ?)""",
                (float(total_gross), float(discount_pct), pm, float(received), float(change), status or "PENDENTE", buyer_cpf),
            )
        elif has_status and not has_bcpf:
            cur = conn.execute(
                """INSERT INTO sales (customer_id, ts, total_gross, discount_pct, payment_method, received, "change", status)
                   VALUES (NULL, datetime('now','localtime'), ?, ?, ?, ?, ?, ?)""",
                (float(total_gross), float(discount_pct), pm, float(received), float(change), status or "PENDENTE"),
            )
        elif not has_status and has_bcpf:
            cur = conn.execute(
                """INSERT INTO sales (customer_id, ts, total_gross, discount_pct, payment_method, received, "change", buyer_cpf)
                   VALUES (NULL, datetime('now','localtime'), ?, ?, ?, ?, ?, ?)""",
                (float(total_gross), float(discount_pct), pm, float(received), float(change), buyer_cpf),
            )
        else:
            cur = conn.execute(
                """INSERT INTO sales (customer_id, ts, total_gross, discount_pct, payment_method, received, "change")
                   VALUES (NULL, datetime('now','localtime'), ?, ?, ?, ?, ?)""",
                (float(total_gross), float(discount_pct), pm, float(received), float(change)),
            )
        return cur.lastrowid

    def _update_sale_status(self, sale_id: int, new_status: str):
        try:
            with self.get_conn() as conn:
                if _sales_has_status(conn):
                    conn.execute("UPDATE sales SET status=? WHERE id=?", (new_status, sale_id))
        except Exception as e:
            print("[PDV] falha ao setar status da venda:", e)

    # ======== CPF opcional (centralizado e em frente ao PDV) ========
    def _ask_buyer_cpf_optional(self) -> str:
        """Abre uma telinha para CPF (opcional). Retorna '' ou CPF validado (somente dígitos, 11)."""
        win = tk.Toplevel(self.root)
        win.title("CPF do comprador (opcional)")
        self._make_modal(win)

        v = tk.StringVar()
        frm = ttk.Frame(win, padding=10); frm.pack(fill="both", expand=True)
        ttk.Label(frm, text="Informe o CPF do comprador (opcional):").grid(row=0, column=0, sticky="w")
        e = ttk.Entry(frm, textvariable=v, width=22); e.grid(row=1, column=0, pady=(4,8), sticky="w")
        msg = ttk.Label(frm, text="Deixe em branco para pular.", foreground="#666")
        msg.grid(row=2, column=0, sticky="w")

        result = {"cpf": ""}

        def ok():
            s = (v.get() or "").strip().replace(".", "").replace("-", "")
            if s and (not s.isdigit() or len(s) != 11):
                messagebox.showerror("CPF", "CPF inválido.\nUse 11 dígitos (apenas números) ou deixe em branco.", parent=win)
                return
            result["cpf"] = s
            win.destroy()

        def skip():
            result["cpf"] = ""
            win.destroy()

        btns = ttk.Frame(frm); btns.grid(row=3, column=0, sticky="e", pady=(10,0))
        ttk.Button(btns, text="Pular", command=skip).pack(side="left", padx=(0,6))
        ttk.Button(btns, text="OK", command=ok).pack(side="left")
        e.focus_set()
        win.bind("<Return>", lambda e: ok())
        win.bind("<Escape>", lambda e: skip())
        win.wait_window()
        self._bring_pdv_front()
        return result["cpf"]

    # ======== Impressão/NFC-e ========
    def _emit_nfce_or_receipt(self, sale_id: int):
        """Tenta emitir NFC-e via nfce_adapter; se não der, imprime recibo TXT."""
        # 1) Tenta NFC-e
        try:
            from nfce_adapter import NFCeAdapter
            try:
                NFCeAdapter(self.get_conn).emit(sale_id)
                return  # sucesso; assume que o adapter cuida da impressão
            except Exception as e:
                print("[NFC-e] Falha ao emitir:", e)
        except Exception:
            pass

        # 2) Recibo TXT
        try:
            with self.get_conn() as conn:
                r_sale = conn.execute(
                    "SELECT id, ts, total_gross, discount_pct, payment_method, received, \"change\", COALESCE(buyer_cpf,'') "
                    "FROM sales WHERE id=?", (sale_id,)
                ).fetchone()
                if not r_sale:
                    raise RuntimeError("Venda não encontrada.")
                items = conn.execute(
                    "SELECT product_id, qty, unit_price FROM sale_items WHERE sale_id=? ORDER BY id", (sale_id,)
                ).fetchall()
                lines = []
                header_name = _get_business_name(conn)  # <<< nome do estabelecimento
                lines.append(f"{header_name} - RECIBO")
                lines.append(f"VENDA #{r_sale[0]}   {r_sale[1]}")
                if r_sale[7]:
                    lines.append(f"CPF: {r_sale[7]}")
                lines.append("-"*38)
                for it in items:
                    pid, qty, unit = it
                    try:
                        p = conn.execute("SELECT sku, name FROM products WHERE id=?", (pid,)).fetchone()
                        sku = p[0] if p else str(pid)
                        nm = p[1] if p else "Produto"
                    except Exception:
                        sku, nm = str(pid), "Produto"
                    total = float(qty) * float(unit)
                    lines.append(f"{sku} {nm}")
                    lines.append(f"   {qty} x R$ {_fmt_money(unit)} = R$ {_fmt_money(total)}")
                lines.append("-"*38)
                disc = float(r_sale[3] or 0.0)
                tot  = float(r_sale[2] or 0.0)
                lines.append(f"Desconto: {disc:.2f}%")
                lines.append(f"TOTAL: R$ {_fmt_money(tot)}")
                lines.append(f"Pagamento: {r_sale[4]}")
                lines.append(f"Recebido: R$ {_fmt_money(float(r_sale[5] or 0.0))}")
                lines.append(f"Troco:    R$ {_fmt_money(float(r_sale[6] or 0.0))}")
                lines.append("")
                lines.append("Obrigado pela preferência!")
                content = "\r\n".join(lines)

            # Envia para impressora (Windows) ou salva .txt
            printed = False
            try:
                import win32print, win32ui  # type: ignore
                raise ImportError  # força fallback do Notepad
            except Exception:
                if os.name == "nt":
                    with tempfile.NamedTemporaryFile(delete=False, suffix=".txt", mode="w", encoding="utf-8") as tf:
                        tf.write(content)
                        temp_path = tf.name
                    try:
                        os.startfile(temp_path, "print")  # type: ignore[attr-defined]
                        printed = True
                    except Exception:
                        printed = False
                        messagebox.showinfo("Recibo", f"Recibo salvo em:\n{temp_path}\nImprima manualmente.", parent=self.root)
                else:
                    with tempfile.NamedTemporaryFile(delete=False, suffix=".txt", mode="w", encoding="utf-8") as tf:
                        tf.write(content)
                        temp_path = tf.name
                    messagebox.showinfo("Recibo", f"Recibo salvo em:\n{temp_path}", parent=self.root)
                    printed = True
            if printed:
                print("[RECIBO] enviado/impressão solicitada.")
        except Exception as e:
            messagebox.showerror("Recibo", f"Falha ao gerar/printar recibo:\n{e}", parent=self.root)
        self._bring_pdv_front()

    def finalize_sale(self):
        # Bloqueio por taxas do dia (opcional)
        try:
            if _fees_guard_block(self.app.db_path, parent_window=self.root):  # type: ignore[attr-defined]
                return
        except Exception:
            try:
                if _fees_guard_block(self.get_conn().__self__.database):  # type: ignore
                    return
            except Exception:
                pass

        # Exige caixa aberto (se existir tabela)
        try:
            with self.get_conn() as conn:
                r = conn.execute("SELECT 1 FROM cash_registers WHERE closed_at IS NULL LIMIT 1").fetchone()
            if not r:
                messagebox.showwarning("PDV", "Abra o caixa antes de registrar vendas (Módulos ▸ Caixa).", parent=self.root)
                self._bring_pdv_front()
                return
        except Exception:
            pass

        if not self.cart:
            messagebox.showwarning("PDV", "Nenhum item no carrinho.", parent=self.root)
            self._bring_pdv_front()
            return

        # CPF opcional antes de gravar venda
        buyer_cpf = self._ask_buyer_cpf_optional()

        subtotal = self._cart_subtotal()
        discount_pct = float(self.var_discount.get() or 0.0)
        total_gross = subtotal * (1.0 - max(0.0, min(discount_pct, 100.0))/100.0)
        pm_raw = self.var_pm.get()
        pm = normalize_payment_method(pm_raw)
        received = _money_to_float(self.var_received.get())
        change = _money_to_float(self.var_change.get())

        # Em PIX, usuário não finaliza manualmente
        if pm == "pix":
            messagebox.showwarning("PIX", "Para PIX use 'Gerar cobrança PIX' e aguarde a confirmação.", parent=self.root)
            self._bring_pdv_front()
            return

        try:
            with self.get_conn() as conn:
                ensure_sales_schema(conn)
                status_init = "PAGA"  # cash/card finalizam na hora
                sale_id = self._insert_sale(conn, status=status_init, pm=pm,
                                            total_gross=total_gross, discount_pct=discount_pct,
                                            received=received, change=change, buyer_cpf=buyer_cpf)
                for it in self.cart:
                    conn.execute(
                        "INSERT INTO sale_items (sale_id, product_id, qty, unit_price) VALUES (?,?,?,?)",
                        (sale_id, it["product_id"], int(it["qty"]), float(it["unit_price"]))
                    )
                    decrement_stock_if_available(conn, it["product_id"], int(it["qty"]))

                # Cria taxa PENDENTE
                try:
                    ensure_tax_table(conn)
                    insert_fee(conn, sale_id=sale_id, status="PENDENTE", amount=0.10)
                except Exception as e:
                    print("[TAXAS] falha ao registrar taxa:", e)

            try:
                self.app.root.event_generate("<<SalesChanged>>", when="tail")
            except Exception:
                pass

            # tentar emitir NFC-e; se não, recibo
            try:
                self._emit_nfce_or_receipt(sale_id)
            except Exception as e:
                print("[PDV] impressão na finalização falhou:", e)

            messagebox.showinfo("PDV", "Venda #{0} registrado com sucesso.".format(sale_id), parent=self.root)
            self._clear_cart()

        except Exception as e:
            messagebox.showerror("PDV", "Falha ao registrar venda:\n{0}".format(e), parent=self.root)
        self._bring_pdv_front()

    # ===== Baixa manual (pendentes) =====
    def _confirm_payment_dialog(self):
        win = tk.Toplevel(self.root)
        win.title("Confirmar pagamento")
        self._make_modal(win)
        win.resizable(False, False)

        frm = ttk.Frame(win, padding=10); frm.pack(fill="both", expand=True)
        ttk.Label(frm, text="Use uma das opções abaixo para baixar vendas pendentes.").grid(row=0, column=0, columnspan=3, sticky="w", pady=(0,8))

        # Última pendente
        last_id = None
        txt = ""
        try:
            with self.get_conn() as conn:
                r = conn.execute(
                    "SELECT id, payment_method, total_gross, ts FROM sales WHERE status='PENDENTE' ORDER BY id DESC LIMIT 1"
                ).fetchone()
            if r:
                last_id = int(r[0])
                txt = "Última pendente: #{0}  ({1})  R$ {2:.2f}  {3}".format(r[0], r[1], float(r[2] or 0), r[3])
            else:
                txt = "Não há vendas pendentes."
        except Exception:
            txt = "Falha ao consultar pendentes."

        ttk.Label(frm, text=txt).grid(row=1, column=0, columnspan=3, sticky="w")

        def do_mark_last():
            if not last_id:
                messagebox.showwarning("Confirmar", "Não há última venda pendente.", parent=win); return
            self._mark_sale_paid(last_id, parent=win)

        ttk.Button(frm, text="Baixar última pendente", command=do_mark_last).grid(row=2, column=0, pady=(8,6), sticky="w")

        # Por ID
        ttk.Label(frm, text="Ou informe o ID da venda:").grid(row=3, column=0, sticky="w", pady=(12,2))
        var_id = tk.StringVar()
        ttk.Entry(frm, textvariable=var_id, width=12).grid(row=3, column=1, sticky="w", padx=(6,0))

        def do_mark_by_id():
            s = (var_id.get() or "").strip()
            if not s.isdigit():
                messagebox.showerror("Confirmar", "Informe um ID numérico.", parent=win); return
            self._mark_sale_paid(int(s), parent=win)

        ttk.Button(frm, text="Confirmar por ID", command=do_mark_by_id).grid(row=3, column=2, sticky="w", padx=(8,0))

        ttk.Button(frm, text="Fechar", command=win.destroy).grid(row=4, column=2, sticky="e", pady=(12,0))

    def _mark_sale_paid(self, sale_id: int, parent=None):
        try:
            with self.get_conn() as conn:
                cur = conn.execute("UPDATE sales SET status='PAGA' WHERE id=?", (int(sale_id),))
                try:
                    conn.execute("""
                        UPDATE payments
                           SET status='PAID', updated_at=datetime('now')
                         WHERE sale_id=? AND status!='PAID'
                    """, (int(sale_id),))
                except Exception:
                    pass
                ensure_tax_table(conn)

            if cur.rowcount < 1:
                messagebox.showwarning("Confirmar", "Venda {0} não encontrada.".format(sale_id), parent=parent or self.root)
                self._bring_pdv_front()
                return

            # Após baixa manual, também tenta imprimir algo
            try:
                self._emit_nfce_or_receipt(sale_id)
            except Exception as e:
                print("[PDV] impressão na baixa manual falhou:", e)

            messagebox.showinfo("Confirmar", "Venda {0} marcada como PAGA.".format(sale_id), parent=parent or self.root)
            try:
                self.app.root.event_generate("<<SalesChanged>>", when="tail")
            except Exception:
                pass
        except Exception as e:
            messagebox.showerror("Confirmar", "Falha ao atualizar: {0}".format(e), parent=parent or self.root)
        self._bring_pdv_front()

# ========= API pública =========
def open_pdv(app, get_conn: Callable):
    return PDVWindow(app, get_conn)
